#' @title Relative Stress Index (RSI)
#' @description
#' This function computes Relative Stress Index (RSI) using any traits (like yield) under stress and non-stress conditions. The lower values of RSI indicates greater tolerance. For more details see Fischer and Wood (1979) <doi:10.1071/AR9791001>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMN Optional numeric scalar: mean of \code{YN}. If NULL, computed.
#' @param YMS Optional numeric scalar: mean of \code{YS}. If NULL, computed.
#' @return A list with \code{YMN}, \code{YMS}, and \code{Result} (data frame with \code{Gen}, \code{RSI}).
#'
#' \deqn{RSI = \frac{(YN/YS)}{(YMS/YMN)}}
#'
#' @references Fischer, R.A. and Wood, J.T. (1979). Drought resistance in spring wheat cultivars. III.* Yield associations with morpho-physiological traits. Australian Journal of Agricultural Research, 30(6), 1001-1020.
#' @examples
#' out = RSI(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
RSI <- function(Gen, YN, YS, YMN = NULL, YMS = NULL) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  if (is.null(YMN)) YMN <- mean(YN, na.rm = TRUE)
  if (is.null(YMS)) YMS <- mean(YS, na.rm = TRUE)
  ratio_env <- YMS / YMN
  num <- YN / YS
  out <- num / ratio_env
  bad <- (YS == 0) | (ratio_env == 0) | !is.finite(ratio_env)
  out[bad] <- NA_real_
  Result <- data.frame(Gen = as.character(Gen), RSI = out, row.names = NULL)
  return(list(YMN = YMN, YMS = YMS, Result = Result))
}
