#' @title Flexible Ranking Utility
#' @aliases ranking
#' @description
#' A thin wrapper around base \code{\link[base]{rank}} with support for
#' ascending/descending order, multiple tie strategies, NA placement,
#' and dense ranks.
#'
#' @param v Numeric (or coercible) vector to rank.
#' @param direction Character, one of \code{"asc"} or \code{"desc"}.
#' @param ties Character, tie-breaking: one of
#'   \code{"average"}, \code{"min"}, \code{"max"}, \code{"first"}, \code{"random"}.
#' @param na.last Character, placement of NAs: \code{"keep"}, \code{"bottom"}, or \code{"top"}.
#' @param dense Logical; if \code{TRUE}, returns dense ranks (1,2,3,...) without gaps.
#'
#' @return An integer/numeric vector of ranks, same length as \code{v}.
#' @examples
#' ranking(c(3, 3, 2, NA, 5), direction="asc", ties="min", na.last="bottom")
#' ranking(c(3, 3, 2, 5), direction = "desc", dense = TRUE)
#' Gen=c("G1","G2","G3")
#' YN=c(10,8,5)
#' YS=c(7,5,3)
#' a=STI(Gen, YN, YS) # for instance STI taken here.
#' out=data.frame(a$Result$Gen, a$Result$STI,
#' ranking(a$Result$STI, direction="desc")
#' )
#' print(out)
#' @export
ranking <- function(v, direction = c("asc","desc"),
                     ties = c("average","min","max","first","random"),
                     na.last = c("keep","bottom","top"),
                     dense = FALSE) {
  direction <- match.arg(direction)
  ties <- match.arg(ties)
  na.last <- match.arg(na.last)
  w <- if (direction == "desc") -v else v
  if (dense) {
    u <- sort(unique(w[!is.na(w)]))
    r <- match(w, u)
    if (na.last == "bottom") r[is.na(w)] <- length(u) + 1
    if (na.last == "top")    r[is.na(w)] <- 0
    return(r)
  } else {
    nal <- switch(na.last, keep = "keep", bottom = TRUE, top = FALSE)
    return(rank(w, ties.method = ties, na.last = nal))
  }
}
