#' @title Internal utility: check vector lengths
#' @description
#' Ensures that \code{YN} and \code{YS} have equal, non-zero lengths. If \code{Gen} is
#' provided (not \code{NULL}), its length must also match. Used internally by stress index functions.
#' @param Gen Optional genotype vector. Default \code{NULL}.
#' @param YN Numeric vector of yields in non-stress (normal) environment.
#' @param YS Numeric vector of yields in stress environment.
#' @keywords internal
#' @noRd
.check_lengths <- function(Gen = NULL, YN, YS) {
  # presence
  if (missing(YN) || missing(YS)) {
    stop("Both 'YN' and 'YS' must be provided.", call. = FALSE)
  }

  # types
  if (!is.numeric(YN) || !is.vector(YN)) {
    stop("'YN' must be a numeric vector.", call. = FALSE)
  }
  if (!is.numeric(YS) || !is.vector(YS)) {
    stop("'YS' must be a numeric vector.", call. = FALSE)
  }

  # lengths
  if (length(YN) == 0L || length(YS) == 0L) {
    stop("'YN' and 'YS' must have length > 0.", call. = FALSE)
  }
  if (length(YN) != length(YS)) {
    stop(sprintf("Lengths differ: length(YN) = %d, length(YS) = %d.",
                 length(YN), length(YS)), call. = FALSE)
  }

  # optional Gen length check
  if (!is.null(Gen) && length(Gen) != length(YN)) {
    stop(sprintf("Length mismatch: length(Gen) = %d, expected %d to match 'YN'/'YS'.",
                 length(Gen), length(YN)), call. = FALSE)
  }

  invisible(TRUE)
}
