% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IPWE_Qopt_DepCen_trt.R
\name{IPWE_Qopt_DepCen_trt}
\alias{IPWE_Qopt_DepCen_trt}
\title{Estimate the Quantile-opt Treatment Regime under the assumption that the censoring
time's distribution only depends on treatment level}
\usage{
IPWE_Qopt_DepCen_trt(data, regimeClass, tau, moPropen = "BinaryRandom",
  cluster = FALSE, p_level = 1, s.tol = 1e-04, it.num = 8,
  pop.size = 6000)
}
\arguments{
\item{data}{raw data.frame}

\item{regimeClass}{a formula specifying the class of treatment regimes to search,
e.g. if \code{regimeClass = a~x1+x2}, and then this function will 
search the class of treatment regimes
of the form 
\deqn{d(x) = I \left(\beta_0 +\beta_1  x_1 + \beta_2  x_2 > 0\right).
}{d(x)=I(\beta_0 +\beta_1 * x1  + \beta_2 * x2 > 0).}
Polynomial arguments are also supported.}

\item{tau}{the quantile of interest}

\item{moPropen}{The propensity score model for the probability of receiving 
treatment level 1.
When \code{moPropen} equals the string "BinaryRandom",  the proportion of observations
receiving treatment level 1 in the sample will be plugged in as an estimate
of the propensity.
Otherwise, this argument should be a formula/string, based on which this function
will fit a logistic regression on the treatment level.  e.g. \code{a1~x1}.}

\item{cluster}{default is FALSE, meaning do not use parallel computing for the genetic algorithm(GA).}

\item{p_level}{choose between 0,1,2,3 to indicate different levels of output
from the genetic function. Specifically, 0 (minimal printing),
  1 (normal), 2 (detailed), and 3 (debug).}

\item{s.tol}{tolerance level for the GA algorithm. This is input for parameter \code{solution.tolerance}
in function \code{rgenoud::genoud}.}

\item{it.num}{the maximum GA iteration number}

\item{pop.size}{an integer with the default set to be 3000. This is roughly the 
number individuals for the first generation
in the genetic algorithm (\code{rgenoud::genoud}).}
}
\description{
Here we assume the censoring variable is independent of covariates 
and potential outcomes 
given the treatment assignment. For example, if evidence shows that patients at 
certain treatment level are prone to experience censoring earlier.
}
\details{
data is a dataframe that contains:
    a(observed treatment assignment), 
    censor_y, and
    delta
}
\examples{
GenerateData_DepCen_trt <- function(n)
{
  x1 <- runif(n, min=-0.5,max=0.5)
  x2 <- runif(n, min=-0.5,max=0.5)
  error <- rnorm(n, sd= 1)
  ph <- exp(-0.5+1*(x1+x2))/(1+exp(-0.5 + 1*(x1+x2)))
  a <- rbinom(n = n, size = 1, prob=ph)
  c <- 1 + 1*a + runif(n = n, min=0, max=2)
   # distribution of `c' depends on treatment level `a'
  cmplt_y <-  pmin(2+x1+x2 +  a*(1 - x1 - x2) +  (0.2 + a*(1+x1+x2)) * error, 4.4)
  censor_y <- pmin(cmplt_y, c)
  delta <- as.numeric(c > cmplt_y)
  return(data.frame(x1=x1,x2=x2,a=a, censor_y = censor_y, delta=delta))
}
\dontshow{
data <- GenerateData_DepCen_trt(50)
fit2 <- IPWE_Qopt_DepCen_trt(data = data, regimeClass = a~x1+x2, moPropen = a~x1+x2,
                                 tau = 0.2, pop.size=300, it.num = 3)
}                        
\donttest{
n <- 400
data <- GenerateData_DepCen_trt(n)
fit1 <- IPWE_Qopt_DepCen_trt(data = data, regimeClass = a~x1+x2, moPropen = a~x1+x2,
                                 tau = 0.2)
                                 }
   
}
