% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simil_DTW_metric.R
\name{simil_DTW_metric}
\alias{simil_DTW_metric}
\title{Similarity metric using Dynamic Time Warping (DTW)}
\usage{
simil_DTW_metric(data, test = NULL, sim = NULL, superposition = NULL)
}
\arguments{
\item{data}{A \code{track} R object, which is a list consisting of two elements:
\itemize{
\item \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
\item \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
}}

\item{test}{Logical; if \code{TRUE}, the function compares the observed DTW distances against
simulated trajectories and calculates \emph{p}-values. Default is \code{FALSE}.}

\item{sim}{A \code{track simulation} R object consisting of a list of simulated trajectories to use for comparison when \code{test = TRUE}.}

\item{superposition}{A character string indicating the method used to align trajectories.
Options are \code{"None"}, \code{"Centroid"}, or \code{"Origin"}. Default is \code{"None"}.}
}
\value{
A \code{track similarity} R object consisting ofa list containing the following elements:
\item{DTW_distance_metric}{A matrix containing the pairwise DTW distances between trajectories.}
\item{DTW_distance_metric_p_values}{(If \code{test} is \code{TRUE}) A matrix containing the \emph{p}-values for the pairwise DTW distances.}
\item{DTW_metric_p_values_combined}{(If \code{test} is \code{TRUE)} The overall \emph{p}-value for the combined DTW distances.}
\item{DTW_distance_metric_simulations}{(If \code{test} is \code{TRUE)} A list of DTW distance matrices from each simulated dataset.}
}
\description{
\code{simil_DTW_metric()}  computes similarity metrics between two or more trajectories using
Dynamic Time Warping (DTW). It allows for different superposition methods
to align trajectories before calculating the DTW metric. The function also supports
testing with simulations to calculate \emph{p}-values for the DTW distance metrics.
}
\details{
The \code{simil_DTW_metric()} function calculates the similarity between trajectories using
the Dynamic Time Warping (DTW) algorithm from the \pkg{dtw} package. The \code{dtw()} function
is used with the \code{dist.method} argument set to \code{"Euclidean"} for computing the local distances
between points in the trajectories.

DTW aligns two time series by minimizing the cumulative distance between their points, creating an optimal alignment despite
variations in length or temporal distortions. The algorithm constructs a distance matrix where each element represents the cost of aligning
points between the two series and finds a warping path through this matrix that minimizes the total distance. The warping path is contiguous and monotonic,
starting from the bottom-left corner and ending at the top-right corner (Cleasby et al., 2019).

DTW measures are non-negative and unbounded, with larger values indicating greater dissimilarity between the time series. This method has been used in various contexts, including ecological studies to analyze and cluster trajectory data (Cleasby et al., 2019).

Potential limitations and biases of DTW include sensitivity to noise and outliers, computational complexity, and the need for appropriate distance metrics.
Additionally, DTW may not always account for all structural differences between trajectories and can be biased by the chosen alignment constraints.
While DTW can handle trajectories of different lengths due to its elastic nature, having trajectories of similar lengths can
improve the accuracy and interpretability of the similarity measure. Similar lengths result in a more meaningful alignment and
can make the computation more efficient. When trajectories differ significantly in length, preprocessing or normalization might
be necessary, and careful analysis is required to understand the alignment path. The function’s flexibility in handling different
lengths allows it to be applied in various contexts. However, large differences in trajectory lengths might introduce potential biases that should be
considered when interpreting the results.

The function offers three different superposition methods to align the trajectories
before \code{DTW()} is applied:
\itemize{
\item \code{"None"}: No superposition is applied.
\item \code{"Centroid"}: Trajectories are shifted to align based on their centroids.
\item \code{"Origin"}: Trajectories are shifted to align based on their starting point.
}

If \code{test = TRUE}, the function can compute \emph{p}-values by comparing the observed DTW
distances with those generated from a set of simulated trajectories. The \emph{p}-values
are calculated for both individual trajectory pairs and for the entire set of trajectories.
}
\section{Logo}{

\if{html}{\figure{Logo.png}{options: width=30\%}}
}

\examples{
# Example 1: Simulating tracks using the "Directed" model and comparing DTW distance
# in the PaluxyRiver dataset
s1 <- simulate_track(PaluxyRiver, nsim = 3, model = "Directed")
simil_DTW_metric(PaluxyRiver, test = TRUE, sim = s1, superposition = "None")

# Example 2: Simulating tracks using the "Constrained" model and comparing DTW distance
# in the PaluxyRiver dataset
s2 <- simulate_track(PaluxyRiver, nsim = 3, model = "Constrained")
simil_DTW_metric(PaluxyRiver, test = TRUE, sim = s2, superposition = "None")

# Example 3: Simulating tracks using the "Unconstrained" model and comparing DTW distance
# in the PaluxyRiver dataset
s3 <- simulate_track(PaluxyRiver, nsim = 3, model = "Unconstrained")
simil_DTW_metric(PaluxyRiver, test = TRUE, sim = s3, superposition = "None")

# Example 4: Simulating and comparing DTW distance in the MountTom dataset using the
# "Centroid" superposition method
sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
s4 <- simulate_track(sbMountTom, nsim = 3)
simil_DTW_metric(sbMountTom, test = TRUE, sim = s4, superposition = "Centroid")

# Example 5: Simulating and comparing DTW distance in the MountTom dataset using the
# "Origin" superposition method
sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
s5 <- simulate_track(sbMountTom, nsim = 3)
simil_DTW_metric(sbMountTom, test = TRUE, sim = s5, superposition = "Origin")

}
\references{
Cleasby, I. R., Wakefield, E. D., Morrissey, B. J., Bodey, T. W., Votier, S. C., Bearhop, S., & Hamer, K. C. (2019). Using time-series similarity measures to compare animal movement trajectories in ecology. Behavioral Ecology and Sociobiology, 73, 1-19.
}
\seealso{
\code{\link{tps_to_track}}, \code{\link{simulate_track}}, \code{\link[dtw]{dtw}}
}
\author{
Humberto G. Ferrón

humberto.ferron@uv.es

Macroevolution and Functional Morphology Research Group (www.macrofun.es)

Cavanilles Institute of Biodiversity and Evolutionary Biology

Calle Catedrático José Beltrán Martínez, nº 2

46980 Paterna - Valencia - Spain

Phone: +34 (9635) 44477
}
