% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RestRserve-package.R
\docType{package}
\name{RestRserve-package}
\alias{RestRserve}
\alias{RestRserve-package}
\title{RestRserve: A Framework for Building HTTP API}
\description{
\if{html}{\figure{logo.png}{options: style='float: right' alt='logo' width='120'}}

Allows to easily create high-performance full featured HTTP APIs from R functions. Provides high-level classes such as 'Request', 'Response', 'Application', 'Middleware' in order to streamline server side application development. Out of the box allows to serve requests using 'Rserve' package, but flexible enough to integrate with other HTTP servers such as 'httpuv'.
}
\details{
\subsection{Introduction}{

Suppose you’ve developed a very useful algorithm or statistical model
and you need to integrate it with some external system. Nowadays HTTP
became de facto a lingua-franca for this kind of tasks.

In this article we will demonstrate how to use RestRserve to build a
basic REST API.
}

\subsection{Workflow overview}{

Generally RestRserve workflow consists of several major steps:
\enumerate{
\item Create application with \code{Application$new()}
\item Create a function which follows RestRserve API:
\itemize{
\item should take 2 arguments - \code{request} and \code{response} as an input.
\code{request} and \code{response} are instances of \code{RestRserve::Request}
and \code{RestRserve::Response}. It is \strong{important to remember} that
both \code{request} and \code{response} are \strong{mutable} objects.
\item should modify \code{response} in place or \code{raise()} exception in case
of error
}
\item Register this function as a handler for an endpoint
\item Start application
}
}

\subsection{1. Create application}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(RestRserve)
app = Application$new()
}\if{html}{\out{</div>}}
}

\subsection{2. Define logic}{

For simplicity we will use Fibonacci number calculation as an algorithm
we want to expose.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{calc_fib = function(n) \{
  if (n < 0L) stop("n should be >= 0")
  if (n == 0L) return(0L)
  if (n == 1L || n == 2L) return(1L)
  x = rep(1L, n)
  
  for (i in 3L:n) \{
   x[[i]] = x[[i - 1]] + x[[i - 2]] 
  \}
  
  return(x[[n]])
\}
}\if{html}{\out{</div>}}

Create function which will handle requests.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fib_handler = function(.req, .res) \{
  n = as.integer(.req$parameters_query[["n"]])
  if (length(n) == 0L || is.na(n)) \{
    raise(HTTPError$bad_request())
  \}
  .res$set_body(as.character(calc_fib(n)))
  .res$set_content_type("text/plain")
\}
}\if{html}{\out{</div>}}

You may have noticed strange \code{.req} and \code{.res} argument names. Starting
from \code{RestRserve} v0.4.0 these “reserved” names allows to benefit from
autocomplete:

<img
src=“\url{https://s3.eu-west-1.amazonaws.com/cdn.rexy.ai/assets/req-res.gif}”
width=“640” style=“vertical-align:bottom”, alt=“request-response
autocomplete gif”>

Technically \code{.req} and \code{.res} are just empty instances of \code{?Request} and
\code{?Response} classes exported by \code{RestRserve} in order to make
autocomplete work.
}

\subsection{2. Register endpoint}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{app$add_get(path = "/fib", FUN = fib_handler)
}\if{html}{\out{</div>}}
}

\subsection{3. Test endpoints}{

Now we can test our application without starting it:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{request = Request$new(path = "/fib", parameters_query = list(n = 10))
response = app$process_request(request)

cat("Response status:", response$status)
#> Response status: 200 OK
cat("Response body:", response$body)
#> Response body: 55
}\if{html}{\out{</div>}}

It is generally a good idea to write unit tests against application. One
can use a common framework such as
\href{https://cran.r-project.org/package=tinytest}{tinytest}.
}

\subsection{4. Add OpenAPI description and Swagger UI}{

Generally it is a good idea to provide documentation along with the API.
Convenient way to do that is to supply a \href{https://swagger.io/docs/specification/about/}{openapi specification}. This as
simple as adding a yaml file as an additional endpoint:

\if{html}{\out{<div class="sourceCode yaml">}}\preformatted{openapi: 3.0.1
info:
  title: RestRserve OpenAPI
  version: '1.0'
servers:
  - url: /
paths:
  /fib:
    get:
      description: Calculates Fibonacci number
      parameters:
        - name: "n"
          description: "x for Fibonnacci number"
          in: query
          schema:
            type: integer
          example: 10
          required: true
      responses:
        200:
          description: API response
          content:
            text/plain:
              schema:
                type: string
                example: 5
        400:
          description: Bad Request
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{yaml_file = system.file("examples", "openapi", "openapi.yaml", package = "RestRserve")
app$add_openapi(path = "/openapi.yaml", file_path = yaml_file)
app$add_swagger_ui(path = "/doc", path_openapi = "/openapi.yaml", use_cdn = TRUE)
}\if{html}{\out{</div>}}
}

\subsection{5. Start the app}{

Now all is ready and we can start application with Rserve backend. It
will block R session and start listening for incoming requests.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{backend = BackendRserve$new()
backend$start(app, http_port = 8080)
}\if{html}{\out{</div>}}
}

\subsection{6. Test it}{

Send request to calculate fibonacci number:

\if{html}{\out{<div class="sourceCode bash">}}\preformatted{curl localhost:8080/fib?n=10
}\if{html}{\out{</div>}}

Check out a swagger UI in the browser: \verb{http://localhost:8080/doc}
}
}
\seealso{
Useful links:
\itemize{
  \item \url{https://restrserve.org}
  \item \url{https://github.com/rexyai/RestRserve}
  \item Report bugs at \url{https://github.com/rexyai/RestRserve/issues}
}

}
\author{
\strong{Maintainer}: Dmitry Selivanov \email{selivanov.dmitriy@gmail.com} (\href{https://orcid.org/0000-0001-5413-1506}{ORCID})

Authors:
\itemize{
  \item Artem Klevtsov \email{a.a.klevtsov@gmail.com} (\href{https://orcid.org/0000-0003-0492-6647}{ORCID})
}

Other contributors:
\itemize{
  \item David Zimmermann \email{david_j_zimmermann@hotmail.com} [contributor]
  \item rexy.ai [copyright holder, funder]
}

}
\keyword{internal}
