#' @title Robust Panel Clustering Algorithm  
#'
#' @description
#' Robust Panel Clustering Algorithm 
#' @details
#' \code{RobPC} function implements the traditional or robust panel clustering analysis without being affected by outliers in the panel data. 
#' 
#' @importFrom stats kmeans prcomp
#' @importFrom trimcluster trimkmeans 
#' @param data the panel data. It must be array.
#' @param k the number of clusters.
#' @param robclust a logical arguments. If robclust=TRUE, the function implements the robust panel clustering analysis. Otherwise, it implements the traditional panel clustering analysis. The default value is robclust=TRUE.
#' @param trim numeric between 0 and 1. Proportion of points to be trimmed. The default value is 0.25.  
#'
#' @export
#' @return a list with 2 elements:
#' \item{clusters}{integer vector coding cluster membership. If robclust=TRUE, this vector includes codes as k+1 to define outliers clusters.}
#' \item{clustering_method}{The used clustering method which is "Robust Panel Clustering" or "Panel Clustering".}
#' @references Bulut et al. (Unpublished). A Robust Clustering Algorithms for Panel Data. 
#' @references Wang, W., & Lu, Y. (2021). Application of clustering analysis of panel data in economic and social research based on R software. Acad. J. Bus. Manag, 3, 98-104.
#' @references Cuesta-Albertos, J. A., Gordaliza, A., & Matrán, C. (1997). Trimmed k-means: an attempt to robustify quantizers. The Annals of Statistics, 25(2), 553-576.
#' @author Hasan BULUT <hasan.bulut@omu.edu.tr>
#' @examples
#' 
#'  set.seed(123)
#'  n_obs <- 10  # Number of observations (N)
#'  n_time <- 5   # Number of time periods (T)
#'  n_vars <- 3   # Number of variables (D)
#'  
#'  data <- array(rnorm(n_obs * n_time * n_vars), dim = c(n_obs, n_time, n_vars))
#'
#'  # Apply the Classical Panel Clustering
#'  
#'     result_classic <- RobPC(data,k=3,robclust = FALSE)
#'     result_classic
#'     
#'  # Apply the Robust Panel Clustering
#'  
#'     result_robust<- RobPC(data,k=3,robclust = TRUE,trim=0.2)
#'     result_robust


RobPC<-function(data, k ,robclust = FALSE, trim = 0.25) {
  # Extract dimensions from data
  n_obs <- dim(data)[1]   # Number of observations (N)
  n_time <- dim(data)[2]  # Number of time periods (T)
  n_vars <- dim(data)[3]  # Number of variables (D)
  
  # Convert data to 2D from 3D
  data_reshaped <- do.call(cbind, lapply(1:dim(data)[3], function(j) as.vector(data[,,j])))
  
  # Standardize data
  data_standardized <- scale(data_reshaped)
  
  # Implement principal component analysis (PCA)
  pca_result <- stats::prcomp(data_standardized, center = TRUE, scale. = TRUE)
  principal_scores <- pca_result$x[, 1]  # İlk bileşeni al
  
  # Convert the PCA scores to the original panel format (N x T)
  reduced_data <- matrix(principal_scores, nrow = n_obs, ncol = n_time, byrow = TRUE)
  
  # Cluster data
  if (robclust) {
    # Trimmed K-Means
    clustering_result <- trimcluster::trimkmeans(reduced_data, k = k, trim = trim)
    cluster_assignments <- clustering_result$classification
    centers <- clustering_result$centers
  } else {
    # K-Means
    clustering_result <- stats::kmeans(reduced_data, centers = k, nstart = 25)
    cluster_assignments <- clustering_result$cluster
    centers <- clustering_result$centers
  }
  
  return(list(
    clusters = cluster_assignments,
    clustering_method = ifelse(robclust, "Robust Panel Clustering", "Panel Clustering")
    ))
}
