#' Plot the SCF Convergence Process for a Single Round of Optimization
#'
#' An internal function plots the generated single-round SCF convergence tibble
#' @param SCFData The tibble generated by SingleRoundOptiSCFIntegratedMonitor() describing the SCF convergence process for single round of optimization(or other Gaussian job types).
#' @param SCFconver A numeric vector showing the SCF convergence requirement read from the gaussian .log file.
#' @returns No return value, called for side effects
#' @importFrom tidyr pivot_longer
#' @importFrom dplyr mutate
#' @importFrom stringr str_replace
#' @importFrom ggplot2 ggplot aes geom_line geom_hline theme_minimal
#' @export
#' @examples
#' library(dplyr)
#' library(stringr)
#' library(ggplot2)
#' library(tidyr)
#'
#' temp_dat <- OptiSCFMonitorAsList(SCFMonitorExample())
#' SingleSCFplotting(temp_dat[[1]][[5]], SCFconver = -log10(temp_dat[[2]]))
#'
#' @name SingleSCFplotting

utils::globalVariables(names = c("cycle", "valueT", "value", "SCFdatType"), package = "SCFMonitor")

SingleSCFplotting <- function(SCFData, SCFconver) {
  SCFData <- SCFData %>%
    tidyr::pivot_longer(!cycle, names_to = "SCFdatType", values_to = "valueT") %>%
    dplyr::mutate(value = as.numeric(stringr::str_replace(valueT, "D", "E")))

  ggplot2::ggplot(
    data = SCFData,
    mapping = ggplot2::aes(
      x = cycle,
      y = -log10(abs(value)),
      color = SCFdatType
    )
  ) +
    ggplot2::geom_line() +
    ggplot2::geom_hline(
      yintercept = SCFconver - 2 + 0.03,
      color = "#F8766D",
      linewidth = 1.05
    ) +
    ggplot2::geom_hline(yintercept = SCFconver - 2, color = "#00BA38") +
    ggplot2::geom_hline(yintercept = SCFconver, color = "#C77CFF") +
    ggplot2::theme_minimal()
}
