% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SMMA.R
\name{SMMA}
\alias{SMMA}
\alias{softmaximin}
\alias{pga}
\title{Soft Maximin Estimation for Large Scale Array Data with Known Groups}
\usage{
softmaximin(X,
            Y,
            zeta,
            penalty = c("lasso", "scad"),
            alg = c("npg", "fista"),
            nlambda = 30,
            lambda.min.ratio = 1e-04,
            lambda = NULL,
            penalty.factor = NULL,
            reltol = 1e-05,
            maxiter = 15000,
            steps = 1,
            btmax = 100,
            c = 0.0001,
            tau = 2,
            M = 4,
            nu = 1,
            Lmin = 0,
            log = TRUE)
}
\arguments{
\item{X}{list containing the Kronecker components (1, 2 or 3) of the Kronecker design matrix.
These are  matrices of sizes \eqn{n_i \times p_i}.}

\item{Y}{array of size \eqn{n_1 \times\cdots\times n_d \times G} containing the response values.}

\item{zeta}{strictly positive float controlling  the softmaximin approximation accuracy.}

\item{penalty}{string specifying the penalty type. Possible values are \code{"lasso", "scad"}.}

\item{alg}{string specifying the optimization algorithm. Possible values are \code{"npg", "fista"}.}

\item{nlambda}{positive integer giving the number of \code{lambda} values. Used when lambda is not specified.}

\item{lambda.min.ratio}{strictly positive float giving the smallest value for \code{lambda}, as a fraction of
\eqn{\lambda_{max}}; the (data dependent) smallest value for which all coefficients are zero.
Used when lambda is not specified.}

\item{lambda}{sequence of strictly positive floats used  as penalty parameters.}

\item{penalty.factor}{array of size \eqn{p_1 \times \cdots \times p_d} of positive floats. Is multiplied
with each element in \code{lambda} to allow differential penalization on the coefficients.}

\item{reltol}{strictly positive float giving the convergence tolerance for the inner loop.}

\item{maxiter}{positive integer giving the maximum number of  iterations allowed for each \code{lambda}
value, when  summing over all outer iterations for said \code{lambda}.}

\item{steps}{strictly positive integer giving the number of steps used in the multi-step adaptive lasso algorithm for non-convex penalties.
Automatically set to 1 when \code{penalty = "lasso"}.}

\item{btmax}{strictly positive integer giving the maximum number of backtracking steps allowed in each iteration. Default is \code{btmax = 100}.}

\item{c}{strictly positive float used in the NPG algorithm. Default is \code{c = 0.0001}.}

\item{tau}{strictly positive float used to control the stepsize for NPG. Default is \code{tau = 2}.}

\item{M}{positive integer giving the look back for the NPG. Default is \code{M = 4}.}

\item{nu}{strictly positive float used to control the stepsize. A  value less that 1 will decrease
the stepsize and a value larger than one will increase it. Default is \code{nu = 1}.}

\item{Lmin}{non-negative float used by the NPG algorithm to control the stepsize. For the default  \code{Lmin = 0}
the maximum step size is the same as for the FISTA algorithm.}

\item{log}{logical variable indicating whether to use log-loss.  TRUE is default and yields the loss below.}
}
\value{
An object with S3 Class "SMMA".
\item{spec}{A string indicating the array dimension (1, 2 or 3) and the penalty.}
\item{coef}{A \eqn{p_1\cdots p_d \times} \code{nlambda} matrix containing the estimates of
the model coefficients (\code{beta}) for each \code{lambda}-value.}
\item{lambda}{A vector containing the sequence of penalty values used in the estimation procedure.}
\item{Obj}{A matrix containing the objective values for each iteration and each model.}
\item{df}{The number of nonzero coefficients for each value of \code{lambda}.}
\item{dimcoef}{A vector giving the dimension of the model coefficient array \eqn{\beta}.}
\item{dimobs}{A vector giving the dimension of the observation (response) array \code{Y}.}
\item{Iter}{A list with 4 items:
\code{bt_iter}  is total number of backtracking steps performed,
\code{bt_enter} is the number of times the backtracking is initiated,
and \code{iter_mat} is a vector containing the  number of  iterations for each \code{lambda} value
and  \code{iter} is total number of iterations i.e. \code{sum(Iter)}.}
}
\description{
Efficient design matrix free procedure for solving a soft maximin problem for
large scale array-tensor structured models, see  \cite{Lund et al., 2020}.
Currently Lasso and SCAD penalized estimation is implemented.
}
\details{
Following \cite{Lund et al., 2020}  this package solves the optimization problem for a linear
model for heterogeneous \eqn{d}-dimensional array data (\eqn{d=1,2,3}) organized in \eqn{G} known groups,
and with identical tensor structured design matrix \eqn{X} across all groups.  Specifically \eqn{n = \prod_i^d n_i} is the
number of observations in each group, \eqn{Y_g}  the  \eqn{n_1\times \cdots \times n_d} response array
for group \eqn{g \in \{1,\ldots,G\}}, and \eqn{X}  a \eqn{n\times p} design matrix, with tensor structure
 \deqn{X = \bigotimes_{i=1}^d X_i.}
For \eqn{d =1,2,3}, \eqn{X_1,\ldots, X_d} are the marginal \eqn{n_i\times p_i} design matrices (Kronecker components).
Using the GLAM framework  the model equation for group \eqn{g\in \{1,\ldots,G\}} is expressed as
 \deqn{Y_g = \rho(X_d,\rho(X_{d-1},\ldots,\rho(X_1,B_g))) + E_g,}
where \eqn{\rho} is the so called rotated \eqn{H}-transfrom (see  \cite{Currie et al., 2006}),
\eqn{B_g} for each \eqn{g} is a (random) \eqn{p_1\times\cdots\times p_d} parameter array
 and \eqn{E_g}  is  a \eqn{n_1\times \cdots \times n_d} error array.

This package solves the penalized soft maximin problem from \cite{Lund et al., 2020}, given by
\deqn{\min_{\beta}\frac{1}{\zeta}\log\bigg(\sum_{g=1}^G \exp(-\zeta \hat V_g(\beta))\bigg) + \lambda  \Vert\beta\Vert_1, \quad \zeta > 0,\lambda \geq 0}
 for the setup described above. Note that
\deqn{\hat V_g(\beta):=\frac{1}{n}(2\beta^\top X^\top vec(Y_g)-\beta^\top X^\top X\beta),}
 is the empirical explained variance from \cite{Meinshausen and B{u}hlmann, 2015}.  See \cite{Lund et al., 2020} for more details and references.

 For \eqn{d=1,2,3}, using only the marginal matrices \eqn{X_1,X_2,\ldots} (for \eqn{d=1} there is only one marginal), the function \code{softmaximin}
solves the soft maximin problem for a sequence of penalty parameters \eqn{\lambda_{max}>\ldots >\lambda_{min}>0}.

Two optimization algorithms  are implemented, a non-monotone
proximal gradient (NPG) algorithm and a fast iterative soft thresholding algorithm (FISTA).
We note that this package also solves the problem above with the penalty given by the SCAD
 penalty, using the multiple step adaptive lasso procedure to loop over the proximal algorithm.
}
\examples{

##size of example
n1 <- 65; n2 <- 26; n3 <- 13; p1 <- 13; p2 <- 5; p3 <- 4

##marginal design matrices (Kronecker components)
X1 <- matrix(rnorm(n1 * p1), n1, p1)
X2 <- matrix(rnorm(n2 * p2), n2, p2)
X3 <- matrix(rnorm(n3 * p3), n3, p3)
X <- list(X1, X2, X3)

component <- rbinom(p1 * p2 * p3, 1, 0.1)
Beta1 <- array(rnorm(p1 * p2 * p3, 0, 0.1) + component, c(p1 , p2, p3))
mu1 <- RH(X3, RH(X2, RH(X1, Beta1)))
Y1 <- array(rnorm(n1 * n2 * n3), dim = c(n1, n2, n3)) + mu1
Beta2 <- array(rnorm(p1 * p2 * p3, 0, 0.1) + component, c(p1 , p2, p3))
mu2 <- RH(X3, RH(X2, RH(X1, Beta2)))
Y2 <- array(rnorm(n1 * n2 * n3), dim = c(n1, n2, n3)) + mu2
Beta3 <- array(rnorm(p1 * p2 * p3, 0, 0.1) + component, c(p1 , p2, p3))
mu3 <- RH(X3, RH(X2, RH(X1, Beta3)))
Y3 <- array(rnorm(n1 * n2 * n3), dim = c(n1, n2, n3)) + mu3
Beta4 <- array(rnorm(p1 * p2 * p3, 0, 0.1) + component, c(p1 , p2, p3))
mu4 <- RH(X3, RH(X2, RH(X1, Beta4)))
Y4 <- array(rnorm(n1 * n2 * n3), dim = c(n1, n2, n3)) + mu4
Beta5 <- array(rnorm(p1 * p2 * p3, 0, 0.1) + component, c(p1 , p2, p3))
mu5 <- RH(X3, RH(X2, RH(X1, Beta5)))
Y5 <- array(rnorm(n1 * n2 * n3), dim = c(n1, n2, n3)) + mu5

Y <- array(NA, c(dim(Y1), 5))
Y[,,, 1] <- Y1; Y[,,, 2] <- Y2; Y[,,, 3] <- Y3; Y[,,, 4] <- Y4; Y[,,, 5] <- Y5;

fit <- softmaximin(X, Y, zeta = 10, penalty = "lasso", alg = "npg")
Betafit <- fit$coef

modelno <- 15
m <- min(Betafit[ , modelno], c(component))
M <- max(Betafit[ , modelno], c(component))
plot(c(component), type="l", ylim = c(m, M))
lines(Betafit[ , modelno], col = "red")

}
\references{
Lund, A., S. W. Mogensen and N. R. Hansen (2020). Soft Maximin Estimation for Heterogeneous Array Data.
\emph{Preprint}.

Meinshausen, N and P. B{u}hlmann (2015). Maximin effects in inhomogeneous large-scale data.
\emph{The Annals of Statistics}. 43, 4, 1801-1830. url = {https://doi.org/10.1214/15-AOS1325}.

Currie, I. D., M. Durban, and P. H. C. Eilers (2006). Generalized linear
array models with applications to multidimensional smoothing.
\emph{Journal of the Royal Statistical Society. Series B}. 68, 259-280. url = {http://dx.doi.org/10.1111/j.1467-9868.2006.00543.x}.
}
\author{
Adam Lund

Maintainer: Adam Lund, \email{adam.lund@math.ku.dk}
}
\keyword{package}
