% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulated_annealing.R
\name{simulatedAnnealing}
\alias{simulatedAnnealing}
\title{An adaptation of the simulated annealing algorithm for psychometric models.}
\usage{
simulatedAnnealing(
  initialModel,
  originalData,
  maxSteps,
  fitStatistic = "cfi",
  temperature = "linear",
  maximize = TRUE,
  Kirkpatrick = TRUE,
  randomNeighbor = TRUE,
  lavaan.model.specs = list(model.type = "cfa", auto.var = TRUE, estimator = "default",
    ordered = NULL, int.ov.free = TRUE, int.lv.free = FALSE, std.lv = TRUE,
    auto.fix.first = FALSE, auto.fix.single = TRUE, auto.cov.lv.x = TRUE, auto.th = TRUE,
    auto.delta = TRUE, auto.cov.y = TRUE),
  maxChanges = 5,
  restartCriteria = "consecutive",
  maximumConsecutive = 25,
  maxItems = NULL,
  items = NULL,
  bifactor = FALSE,
  setChains = 1,
  shortForm = T,
  ...
)
}
\arguments{
\item{initialModel}{The initial model as a \code{character} vector with lavaan model.syntax.}

\item{originalData}{The original \code{data.frame} with variable names.}

\item{maxSteps}{The number of iterations for which the algorithm will run.}

\item{fitStatistic}{Either a single model fit statistic produced by lavaan, or a user-defined fit statistic function.}

\item{temperature}{Either an acceptable \code{character} value, or a user-defined temperature function. The acceptable values are "linear", "quadratic", or "logistic".}

\item{maximize}{Logical indicating if the goal is to maximize (\code{TRUE}) the fitStatistic for model selection.}

\item{Kirkpatrick}{Either \code{TRUE} to use Kirkpatrick et al. (1983) acceptance probability, or a user-defined function for accepting proposed models.}

\item{randomNeighbor}{Either \code{TRUE} to use the included function for randomNeighbor selection, or a user-defined function for creating random models.}

\item{lavaan.model.specs}{A \code{list} which contains the specifications for the
lavaan model. The default values are the defaults for lavaan to perform a
CFA. See \link[lavaan]{lavaan} for more details.}

\item{maxChanges}{An \code{integer} value greater than 1 setting the maximum number of parameters to change within randomNeighbor. When creating a short form, should be no greater than the smallest reduction in items loading on one factor; e.g., when reducing a 2-factor scale from 10 items on each factor to 8 items on the first and 6 items on the second, maxChanges should be no greater than 2.}

\item{restartCriteria}{Either "consecutive" to restart after maxConsecutiveSelection times with the same model chosen in a row, or a user-defined function.}

\item{maximumConsecutive}{A positive \code{integer} value used with restartCriteria.}

\item{maxItems}{When creating a short form, a \code{vector} of the number of items per factor you want the short form to contain. Defaults to \code{NULL}.}

\item{items}{A \code{character} vector of item names. Defaults to \code{NULL}. Ignored if \code{maxItems==FALSE}.}

\item{bifactor}{Logical. Indicates if the latent model is a bifactor model. If \code{TRUE}, assumes that the last latent variable in the provided model syntax is the bifactor (i.e., all of the retained items will be set to load on the last latent variable). Ignored if \code{maxItems==FALSE}.}

\item{setChains}{Numeric. Sets the number of parallel chains to run. Default to \code{1}, which also sets the algorithm to run serially (e.g., on a single processor). Values greater than \code{1} result in the chains running on parallel processes using the \code{doSNOW} and \code{foreach} packages.}

\item{shortForm}{Logical. Are you creating a short form (\code{TRUE}) or not (\code{FALSE})? Default is \code{TRUE}.}

\item{...}{Further arguments to be passed to other functions. Not implemented for any of the included functions.}
}
\value{
A named list: the 'bestModel' found, the 'bestFit', and 'allFit' values found by the algorithm.
}
\description{
Simulated annealing mimics the physical process of annealing metals together. \href{https://www.science.org/doi/10.1126/science.220.4598.671}{Kirkpatrick et al. (1983)} introduces this analogy and demonstrates its use; the implementation here follows this demonstration closely, with some modifications to make it better suited for psychometric models.
}
\details{
\strong{Outline of the Pieces of the Simulated Annealing Algorithm}
\itemize{
\item initialModel -- the initial, full form
\item currentModel -- the model of the current step
\item maxSteps -- the maximum number of steps (iterations)
\item currentStep -- the current step
\item currentTemp -- the current temperature. A function of the number of steps (such that temp = 0 at maxSteps), and values that control the shape of the overall temperature. A part of the function that determines the acceptance probability of newly -- generated models
\item randomNeighbor -- a function that determines how the form is changed at each step. Should be able to change one or more parameters, and should have a way to control how many are changed.
\item goal -- a function that determines the "goodness" of the currentModel. Typically in SA goodness is defined as minimization! Sometimes called an energy function
\item selectionFunction -- a function that determines if a randomNeighbor change is accepted. Uses the goal function that determines the "goodness" of the currentModel and the "goodness" of the randomNeighbor, and the currentTemp to generate a probability of acceptance, then compares this probability to a Uniform(0,1) variable to determine if accepted or not. A standard version of this is:
\figure{SA-goal.jpg}
(Kirkpatrick et al., 1983)
\item bestModel -- the model with the best value of the goal function achieved so far
\item bestGoal -- the best value of the goal function achieved so far
\item restartCriteria -- if utilized, this would "restart" the SA process by changing currentModel to bestModel and continuing the process. Could be based on (1) the currentStep value, (2) the difference between goal(currentModel) and goal(bestModel), (3) randomness (i.e., could randomly restart, could randomly restart based on some values, etc), (4) other criteria.
}
}
\examples{
\dontrun{
data(exampleAntModel)
data(simulated_test_data)
trial1 <- simulatedAnnealing(
  initialModel = lavaan::cfa(
    model = exampleAntModel,
    data = simulated_test_data
  ),
  originalData = simulated_test_data, maxSteps = 3,
  fitStatistic = "rmsea", maximize = FALSE
)
summary(trial1) # shows the resulting model

trial2 <- simulatedAnnealing(
  initialModel = exampleAntModel,
  originalData = simulated_test_data,
  maxSteps = 2, maxItems = 30, items = paste0("Item", 1:56)
)
summary(trial2) # shows the resulting model
}
}
