% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/test_stats.R
\name{wise_test}
\alias{wise_test}
\title{Conducts the serial independence test (WISE) based on a similarity matrix}
\usage{
wise_test(sim, dependence = "proximity", alpha = 0.05, weight = NULL, h = 4)
}
\arguments{
\item{sim}{an n by n similarity matrix, typically generated from wise_sim().}

\item{dependence}{design for the weight matrix W:
if "proximity", \eqn{W_{ij} =  (1 / (|i - j|^2 + 1))-1};
if "periodicity", then \eqn{W_{ij} = |cos(|i-j|\pi/h)|-1};
If "customized", users should input their self-defined weight matrix through the parameter "weight".
The default is "proximity"}

\item{alpha}{the nominal significance level (default is 0.05).}

\item{weight}{an n by n weight matrix with zero diagonal (only used if dependence = "customized").}

\item{h}{the estimated periodicity (default is 4). The parameter is used only if dependence = "periodicity".}
}
\value{
A list containing:
\item{p_value}{The p-value of the test.}
\item{test_statistic_sq}{The value of the squared test statistic.}
\item{critical_value}{The chi-square critical value at the given significance level.}
\item{t}{The unstandardized test statistic.}
\item{permutation_mean}{The mean of t under the permutation null.}
\item{permutation_variance}{The variance of t under the permutation null.}
}
\description{
Returns the p-value of WISE, the squared test statistic, and related quantities (the chi-square critical value, permutation mean, permutation variance).
}
\examples{
library(MASS)
n <- 100
p <- 50

# Example 1: iid data
set.seed(123)
data_iid <- mvrnorm(n = n, mu = rep(0, p) , Sigma = diag(p))
wise_test(
 wise_sim(data_iid, measure = "distance", metric = "manhattan"),
 dependence = "proximity",
 alpha = 0.05
)

# Example 2: AR(1)
set.seed(123)
data_ar <- matrix(0, nrow = n, ncol = p)
error <- mvrnorm(n = n, mu = rep(0,p), Sigma = diag(p))
data_ar[1,] <- error[1,]
phi <- 0.1 * diag(p)
for (t in 2:n) {
  data_ar[t, ] <- phi \%*\% data_ar[t - 1, ] + error[t,]
}
wise_test(
 wise_sim(data_ar, measure = "distance", metric = "manhattan"),
 dependence = "proximity",
 alpha = 0.05
)

# Example 3: NMA(2)
set.seed(123)
data_nma <- matrix(0, nrow = n, ncol = p)
error <- mvrnorm(n = n, mu = rep(0,p), Sigma = diag(p))
data_nma[1:2, 1:p] <-error[1:2,1:p]
for (i in 3:n) {
  data_nma[i, ] <- error[i,]*error[i-1,]*error[i-2,]
}
wise_test(
 wise_sim(data_nma, measure = "distance", metric = "manhattan"),
 dependence = "proximity",
 alpha = 0.05
)
}
