#' Entropy rate per jump
#' 
#' Computes the entropy rate per jump of the Markov chain generated by the compartmental system
#' 
#' @param A A constant compartmental square matrix with cycling rates in the
#' diagonal and transfer rates in the off-diagonal.
#' @param u A one-column matrix defining the amount of inputs per compartment.
#' @return A scalar value with the entropy rate per jump
#' @references 
#' Metzler, H. (2020). Compartmental systems as Markov chains : age, 
#' transit time, and entropy (T. Oertel-Jaeger, I. Pavlyukevich, and C. Sierra, 
#' Eds.) 
#' [PhD thesis](https://suche.thulb.uni-jena.de/Record/1726091651)
#' @examples
#' B6=matrix(c(-1,1,0,0,-1,1,0,0,-1),3,3); u6=matrix(c(1,0,0))
#' entropyRatePerJump(A=B6, u=u6)
entropyRatePerJump=function(A,u){
  if(dim(A)[1]==dim(A)[2]) d=dim(A)[1] else(stop("A must be a square matrix"))
  beta=u/sum(u)
  z=-colSums(A)
  lambda=-1*diag(A)
  x=-1*solve(A)%*%u
  y=rbind(x,sum(u))
  Q=rbind(cbind(A,beta),c(z, -1))
  DQ=diag(-1*diag(Q))
  DQy=DQ%*%y
  Pi= DQy/sum(DQy)
  A_lambda=(A+diag(lambda))%*%diag(1/lambda)
  z_lambda=z/lambda
  theta1_1=colSums(-A_lambda*log(A_lambda), na.rm=TRUE)
  theta1=sum(Pi[1:d] * rowSums(cbind(theta1_1, -z_lambda*log(z_lambda)), na.rm = TRUE), na.rm = TRUE)
  theta_z=theta1-(Pi[d+1]*sum(beta*log(beta), na.rm = TRUE))
  theta_P_inf=sum(Pi[1:d]*(1-log(lambda))) + theta_z
  return(theta_P_inf)
}

