% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eigensound.R
\name{eigensound}
\alias{eigensound}
\title{Sound waves onto morphometric data}
\usage{
eigensound(
  analysis.type = NULL,
  wav.at = NULL,
  store.at = wav.at,
  dBlevel = 25,
  flim = c(0, 10),
  tlim = c(0, 1),
  x.length = 80,
  y.length = 60,
  log.scale = TRUE,
  back.amp = 35,
  add.points = FALSE,
  add.contour = TRUE,
  lwd = 1,
  EQ = c(0.05, 0.15, 0.3, 0.5, 0.7, 0.85, 0.95),
  mag.time = 1,
  f = 44100,
  wl = 512,
  ovlp = 70,
  plot.exp = TRUE,
  plot.as = "jpeg",
  plot.type = "surface",
  rotate.Xaxis = 60,
  rotate.Yaxis = 40,
  TPS.file = NULL
)
}
\arguments{
\item{analysis.type}{type of analysis intended. If \code{analysis.type = "threeDshape"}, semilandmarks are acquired from spectrogram data using a 3D representation of sound (same as in MacLeod et al., 2013). If \code{analysis.type = "twoDshape"} and \code{add.points = TRUE}, semilandmarks are acquired using energy quantiles and a 2D curve of relative amplitude. By default: \code{analysis.type = NULL} (i.e. method must be specified before the analysis).}

\item{wav.at}{filepath to the folder where \code{".wav"} files are stored. Should be presented between quotation marks. By default: \code{wav.at = NULL} (i.e. user must specify the filepath to \code{".wav"} files)}

\item{store.at}{filepath to the folder where spectrogram plots and \code{tps} file will be stored. Should be presented between quotation marks. By default: \code{store.at = wav.at} (i.e. store outputs in the same folder as \code{".wav"} files)}

\item{dBlevel}{absolute amplitude value to be used as relative amplitude contour, which will serve as reference for semilandmark acquisition in both \code{analysis.type = "threeDshape"} and \code{"twoDshape"}. By default: \code{dBlevel = 25}}

\item{flim}{modifications of the frequency limits (Y-axis). Vector with two values in kHz. By default: \code{flim = c(0, 10)}}

\item{tlim}{modifications of the time limits (X-axis). Vector with two values in seconds. By default: \code{tlim = c(0, 1)}}

\item{x.length}{only applies when \code{analysis.type = "threeDshape"}. Length of sequence (i.e. number of cells per side on sound window) to be used as sampling grid coordinates on the time (X-axis).  By default: \code{x.length = 80}}

\item{y.length}{only applies when \code{analysis.type = "threeDshape"}. Length of sequence (i.e. number of cells per side on sound window) to be used as sampling grid coordinates on the frequency (Y-axis). By default: \code{y.length = 60}}

\item{log.scale}{only applies when \code{analysis.type = "threeDshape"}. A logical. If \code{TRUE}, \code{eigensound} will use a logarithmic scale on the time (X-axis), which is recommeded when the analyzed sounds present great variation on this axis (e.g. emphasize short duration sounds). If \code{FALSE}, a linear scale is used instead (same as MacLeod et al., 2013). By default: \code{log.scale = TRUE}}

\item{back.amp}{only applies when \code{analysis.type = "twoDshape"} and  \code{plot.exp = TRUE}. Absolute amplitude value to be used as background in the 2D spectrogram plot. By default: \code{back.amp = 35}}

\item{add.points}{only applies when \code{analysis.type = "twoDshape"}. A logical. If \code{TRUE}, \code{eigensound} will compute semilandmarks acquired by cross-correlation between energy quantiles (i.e. \code{EQ}) and a curve of relative amplitude (i.e. \code{dBlevel}). If \code{plot.exp = TRUE}, semilandmarks will be included in spectrogram plots. By default: \code{add.points = FALSE} (see \code{Details})}

\item{add.contour}{only applies when \code{analysis.type = "twoDshape"} and  \code{plot.exp = TRUE}. A logical. If \code{TRUE}, exported spectrogram plots will include the curves of relative amplitude at the level specified by \code{dBlevel}. By default: \code{add.contour = TRUE}}

\item{lwd}{only applies when \code{plot.exp = TRUE} and \code{add.contour = TRUE}. The line width for the curves of relative amplitude at the level specified by \code{dBlevel}. Same as in \code{\link{par}}. By default: \code{lwd = 1}}

\item{EQ}{only applies when \code{analysis.type = "twoDshape"} and \code{add.points = TRUE}. A vector of energy quantiles intended (with \code{0 < EQ < 1}). By default: \code{EQ = c(0.05, 0.15, 0.3, 0.5, 0.7, 0.85, 0.95)} \strong{Note:} When dealing with narrow banded calls, consider reducing the number of quantiles to prevent errors in the analysis.}

\item{mag.time}{only applies when \code{analysis.type = "twoDshape"}. Optional argument for magnifying the time coordinates (X-axis). This is sometimes desired for small sound windows (e.g. less than 1 s), in which the time coordinates will be on a different scale than that of frequency coordinates. In those cases, it is recommended to include \code{mag.time = 10} or \code{mag.time = 100}, depending on the lenght of sound window. By default: \code{mag.time = 1} (i.e. no magnification is performed)}

\item{f}{sampling frequency of \code{Wave}'s for the analysis (in Hz). By default: \code{f = 44100}}

\item{wl}{length of the window for the analysis. By default: \code{wl = 512}}

\item{ovlp}{overlap between two successive windows (in \%) for increased spectrogram resolution. By default: \code{ovlp = 70}}

\item{plot.exp}{a logical. If \code{TRUE}, for each \code{".wav"} file on the folder indicated by \code{wav.at}, \code{eigensound} will store a spectrogram image on the folder indicated by \code{store.at}. Depending on the \code{analysis.type}, plots may consist of 2D or 3D spectrogram images. By default: \code{plot.exp = TRUE}}

\item{plot.as}{only applies when \code{plot.exp = TRUE}. \code{plot.as = "jpeg"} will generate compressed images for quick inspection of semilandmarks; \code{plot.as = "tiff"} or \code{"tif"} will generate uncompressed high resolution images that can be edited and used for publication. By default: \code{plot.as = "jpeg"}}

\item{plot.type}{only applies when \code{analysis.type = "threeDshape"} and  \code{plot.exp = TRUE}. \code{plot.type = "surface"} will produce simplified 3D sound surfaces from the calculated semilandmarks (same output employed by MacLeod et al., 2013); \code{plot.type = "points"} will produce 3D graphs with semilandmarks as points. By default: \code{plot.type = "surface"}}

\item{rotate.Xaxis}{only applies when \code{analysis.type = "threeDshape"} and  \code{plot.exp = TRUE}. Rotation of the X-axis. Same as \code{theta} from \code{\link[plot3D]{persp3D}}. By default: \code{rotate.Xaxis = 60}}

\item{rotate.Yaxis}{only applies when \code{analysis.type = "threeDshape"} and  \code{plot.exp = TRUE}. Rotation of the Y-axis. Same as \code{phi} from \code{\link[plot3D]{persp3D}}. By default: \code{rotate.Yaxis = 40}}

\item{TPS.file}{Desired name for the \code{tps} file containing semilandmark coordinates. Should be presented between quotation marks. \strong{Note:} Whenever \code{analysis.type = "twoDshape"}, it will only work if \code{add.points = TRUE}. By default: \code{TPS.file = NULL} (i.e. prevents \code{eigensound} from creating a \code{tps} file)}
}
\description{
\code{eigensound} is the main feature of \code{SoundShape} package. For each \code{".wav"} file on a given folder, the fuction will compute spectrogram data and acquire semilandmarks using a 3D representation of sound (\code{analysis.type = "threeDshape"}), allowing its users to acquire, and simultaneously store point coordinates (i.e. semilandmarks) as an R object, and/or in TPS format – the native file format of James Rohlf’s TPS series (Rohlf, 2015).

Moreover, \code{eigensound} also allow its user to export 2D and 3D spectrogram images (\code{plot.exp = TRUE}) that are helpful during the protocol for error verification and for illustrative purposes (see Rocha & Romano \emph{in prep}). Alternativaly, \code{eigensound} feature the option of acquiring semilandmarks as the cross-correlation between energy quantiles and a curve of relative amplitude from 2D spectrograms (\code{analysis.type = "twoDshape"}; see \code{Details} section).
}
\details{
When \code{analysis.type = "twoDshape"} and \code{add.points = TRUE}, \code{eigensound} will compute semilandmarks acquired by cross-correlation between energy quantiles (i.e. \code{EQ}) and a curve of relative amplitude (i.e. \code{dBlevel}). However, this is often subtle and prone to incur in errors (e.g. bad alignment of acoustic units; inappropriate X and Y coordinates for the sound window; narrow banded calls). Therefore, a more robust protocol of error verification is achieved using \code{add.points = FALSE} and \code{add.contour = TRUE} (default), which allow for quick verification of acoustic units alignment and the shape of each curve of relative amplitude (specified by \code{dBlevel}).
}
\note{
In order to store the results from \code{eigensound} function and proceed with the Geometric Morphometric steps of the analysis (e.g. \code{\link[geomorph]{geomorph}} package; Adams et al., 2017), one can simultaneosly assign the function's output to an \code{R} object and/or store them as a \code{tps} file to be used by numerous softwares of geometric analysis of shape, such as the TPS series (Rohlf, 2015).

Additionally, one may also export 2D or 3D plots as \code{jpeg} (compressed image) or \code{tiff} (uncompressed image) file formats, which can be edited for publication purposes.
}
\examples{
library(seewave)
library(tuneR)

# Create temporary folder to store ".wav" files
wav.at <- file.path(base::tempdir(), "eigensound")
if(!dir.exists(wav.at)) dir.create(wav.at)

# Create temporary folder to store results
store.at <- file.path(base::tempdir(), "eigensound-output")
if(!dir.exists(store.at)) dir.create(store.at)

# Cut acoustic units from original Wave
cut.cuvieri <- cutw(cuvieri, f=44100, from=0, to=0.9, output = "Wave")
cut.centralis <- cutw(centralis, f=44100, from=0, to=0.9, output = "Wave")
cut.kroyeri <- cutw(kroyeri, f=44100, from=0.2, to=1.1, output = "Wave")

# Export ".wav" files containing acoustic units and store on previosly created folder
writeWave(cut.cuvieri, filename = file.path(wav.at, "cut.cuvieri.wav"), extensible = FALSE)
writeWave(cut.centralis, filename = file.path(wav.at, "cut.centralis.wav"), extensible = FALSE)
writeWave(cut.kroyeri, filename = file.path(wav.at, "cut.kroyeri.wav"), extensible = FALSE)


\donttest{
# Create 2D spectrograms using analysis.type = "twoDshape"
eigensound(analysis.type = "twoDshape", flim=c(0, 4), tlim=c(0, 0.8),
           plot.exp=TRUE, wav.at = wav.at, store.at = store.at)

# Create 3D spectrograms using analysis.type = "threeDshape" and store point coordinates
eig.data <- eigensound(analysis.type = "threeDshape", plot.exp=TRUE,
              wav.at = wav.at, store.at = store.at, flim=c(0, 4), tlim=c(0, 0.8))
}

}
\references{
Adams, D. C., M. L. Collyer, A. Kaliontzopoulou & Sherratt, E. (2017) \emph{Geomorph: Software for geometric morphometric analyses}. R package version 3.0.5. https://cran.r-project.org/package=geomorph.

MacLeod, N., Krieger, J. & Jones, K. E. (2013). Geometric morphometric approaches to acoustic signal analysis in mammalian biology. \emph{Hystrix, the Italian Journal of Mammalogy, 24}(1), 110-125.

Rocha, P. & Romano, P. (2021) The shape of sound: A new \code{R} package that crosses the bridge between Bioacoustics and Geometric Morphometrics. \emph{Methods in Ecology and Evolution, 12}(6), 1115-1121.

Rohlf, F.J. (2015) The tps series of software. \emph{Hystrix 26}, 9-12.
}
\seealso{
\code{\link{align.wave}}, \code{\link[geomorph]{geomorph}}, \code{\link[seewave]{seewave}}

Useful links:
\itemize{
\item{\url{https://github.com/p-rocha/SoundShape}}
\item{Report bugs at \url{https://github.com/p-rocha/SoundShape/issues}}}
}
\author{
Pedro Rocha
}
