% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{WeatherTask}
\alias{WeatherTask}
\title{Weather Task with Priming and Precise and Imprecise Probabilities}
\format{
A data frame with 345 observations on 4 variables:
\describe{
\item{agreement}{numeric. Probability indicated by participants, or the
average between minimum and maximum estimates in the imprecise condition.
Response variable scaled to (0, 1).}
\item{priming}{factor with levels \code{two-fold} (case prime) and \code{seven-fold}
(class prime). Indicates the partition priming condition.}
\item{eliciting}{factor with levels \code{precise} and \code{imprecise} (lower and
upper limit). Indicates whether participants gave point estimates or
interval estimates.}
}
}
\source{
Taken from Smithson et al. (2011) supplements.
}
\usage{
WeatherTask
}
\description{
Data from a cognitive psychology experiment on probabilistic learning and
probability judgments. Participants estimated probabilities for weather events
under different priming and precision conditions.
}
\details{
All participants in the study were either first- or second-year undergraduate
students in psychology, none of whom had a strong background in probability or
were familiar with imprecise probability theories.

\strong{Task description:} Participants were asked: "What is the probability that the
temperature at Canberra airport on Sunday will be higher than 'specified
temperature'?"

\strong{Experimental manipulations:}
\itemize{
\item \strong{Priming:} Two-fold (simple binary: above/below) vs. seven-fold (multiple
temperature categories)
\item \strong{Eliciting:} Precise (single probability estimate) vs. imprecise (lower and
upper bounds)
}

The study examines how partition priming (number of response categories) and
elicitation format affect probability judgments. Classical findings suggest that
more categories (seven-fold) lead to different probability assessments than
binary categories (two-fold).
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(WeatherTask)

# Example 1: Main effects model
# Probability judgments affected by priming and elicitation format
fit_kw <- gkwreg(
  agreement ~ priming + eliciting,
  data = WeatherTask,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Seven-fold priming may shift probability estimates
#   Imprecise elicitation may produce different mean estimates

# Example 2: Interaction model with heteroscedasticity
# Priming effects may differ by elicitation format
# Variability may also depend on conditions
fit_kw_interact <- gkwreg(
  agreement ~ priming * eliciting |
    priming + eliciting,
  data = WeatherTask,
  family = "kw"
)
summary(fit_kw_interact)

# Interpretation:
# - Alpha: Interaction tests if partition priming works differently
#   for precise vs. imprecise probability judgments
# - Beta: Precision varies by experimental condition

# Test interaction
anova(fit_kw, fit_kw_interact)

# Example 3: McDonald distribution for polarized responses
# Probability judgments often show polarization (clustering at extremes)
# particularly under certain priming conditions
fit_mc <- gkwreg(
  agreement ~ priming * eliciting | # gamma
    priming * eliciting | # delta
    priming, # lambda: priming affects polarization
  data = WeatherTask,
  family = "mc",
  control = gkw_control(method = "BFGS", maxit = 1500)
)
summary(fit_mc)

# Interpretation:
# - Lambda varies by priming: Seven-fold priming may produce more
#   extreme/polarized probability judgments
}
}
\references{
Smithson, M., Merkle, E.C., and Verkuilen, J. (2011). Beta Regression Finite
Mixture Models of Polarization and Priming. \emph{Journal of Educational and
Behavioral Statistics}, \strong{36}(6), 804--831.
\doi{10.3102/1076998610396893}

Smithson, M., and Segale, C. (2009). Partition Priming in Judgments of
Imprecise Probabilities. \emph{Journal of Statistical Theory and Practice},
\strong{3}(1), 169--181.
}
\keyword{datasets}
