#' @title Core vertices of games with two, three or four players
#' @description Given a game with no more than four players, this function computes its core vertices.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return If the core of \code{v} is non-empty, the core vertices are returned, as a matrix in which each row is a vertex.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @note Function \code{corevertices} can also compute the core vertices of games with less than five players, but takes a different approach.
#' @examples
#' # 2 players:
#' corevertices234(c(-58,4,13))
#'
#' # 3 players:
#' corevertices234(c(1,5,10,6,11,15,16)) # additive game
#'
#' # 4 players:
#' corevertices234(c(0,0,0,0,4,3,5,2,4,5,10,19,20,30,100)) # convex game
#' corevertices234(c(0,0,0,0,1,2,1,1,1,1,4,3,2,1,7)) # not convex game
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # cost game
#' -corevertices234(-cost.v) # core vertices of the cost game
#' @seealso \link{balancedcheck}, \link{corevertices}, \link{plotcoreset},
#' @export

corevertices234 <- function(v, binary = FALSE) {

  ##################################
  ### Comprobación datos entrada ###
  ##################################

  nC <- length(v)
  n <- log(nC + 1) / log(2)

  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (n > 4) {
    stop("'v' cannot be a game of more than four players.")
  }

  # CUIDADO: ESTA FUNCIÓN ES ESPECIAL, HACE LAS CUENTAS EN LEXICOGRÁFICO.
  if (binary == TRUE) { # Si el juego se introdujo en binario, lo pasamos a lexicográfico.
    v <- bin2lex(v)
  }

  ################################
  ###### Cuerpo de la función ####
  ################################

  if (n == 2) {  # Si hay 2 jugadores, el conjunto de imputaciones coincide con el core.
    if (v[3] < v[1] + v[2]) {
      message("The core of 'v' is empty.")
      return(NULL)
    } else { # Si el juego es equilibrado...
      m <- matrix(nrow = 2, ncol = 2)
      m[1,] <- c(v[1], v[3] - v[1])
      m[2,] <- c(v[3] - v[2], v[2])
      vertices <- unique(m, MARGIN = 1)
    }
    return(vertices)
  }

  if (n == 3) {
    # Comprobación de si el juego es equilibrado
    if (v[7] < v[1] + v[2] + v[3] | v[7] < v[1] + v[6] | v[7] < v[2] + v[5] | v[7] < v[3] + v[4] | v[7] < 1/2 * (v[4] + v[5] + v[6])) {
      message(paste("The core of", deparse(substitute(v)), "is empty."))
      return(NULL)
    } # Cierre del bloque if

    # Cálculo de la envolvente exacta (para 3 es sencillo). Como todo juego de 3 exacto es convexo, luego calculamos los vectores de contribuciones marginales
    t <- c(v[1:3], max(v[4], v[1] + v[2]), max(v[5], v[1] + v[3]), max(v[6], v[2] + v[3]), v[7])
    w <- c(max(t[1], t[4] + t[5] - t[7]), max(t[2], t[4] + t[6] - t[7]), max(t[3], t[5] + t[6] - t[7]), t[4:7])

    # Vértices del núcleo:
    ## Vectores de contribuciones marginales
    m <- matrix(nrow = 6, ncol = 3)
    m[1, ] <- c(w[1], w[4] - w[1], w[7] - w[4])
    m[2, ] <- c(w[1], w[7] - w[5], w[5] - w[1])
    m[3, ] <- c(w[4] - w[2], w[2], w[7] - w[4])
    m[4, ] <- c(w[7] - w[6], w[2], w[6] - w[2])
    m[5, ] <- c(w[5] - w[3], w[7] - w[5], w[3])
    m[6, ] <- c(w[7] - w[6], w[6] - w[3], w[3])

    vertices <- unique(m, MARGIN = 1)
    return(vertices)
  } # Cierre del bloque if

  if (n == 4) {
    v1=v[1];v2=v[2];v3=v[3];v4=v[4];
    v12=v[5];v13=v[6];v14=v[7];v23=v[8];v24=v[9];v34=v[10];
    v123=v[11];v124=v[12];v134=v[13];v234=v[14];
    v1234=v[15];

    # Comprobación de si el juego es equilibrado
    if (v1234 < (v1 + v2 + v3 + v4) | v1234 < (v1 + v234) | v1234 < (v1 + v2 + v3 + v4) | v1234 < (v1 + v234) | v1234 < (v2 + v134) | v1234 < (v3 + v124) | v1234 < (v4 + v123) |
        v1234 < (v1 + v2 + v34) | v1234 < (v1 + v3 + v24) | v1234 < (v1 + v4 + v23) | v1234 < (v2 + v3 + v14)| v1234 < (v2 + v4 + v13) | v1234 < (v3 + v4 + v12) |
        v1234 < (v12 + v34) | v1234 < (v13 + v24) | v1234 < (v14 + v23) |
        v1234 < 1/3 * (v12 + v13 + v14 + v23 + v24 + v34) |
        v1234 < 1/2 * (v23 + v24 + v34) + v1 | v1234 < 1/2 * (v13 + v14 + v34) + v2 | v1234 < 1/2 * (v12 + v14 + v24) + v3 | v1234 < 1/2 * (v12 + v13 + v23) + v4 |
        v1234 < 1/3 * (v123 + v124 + v134 + v234) |
        v1234 < (1/3 * (v12 + v13 + v14) + 2/3 * v234) | v1234 < (1/3 * (v12 + v23 + v24) + 2/3 * v134) |
        v1234 < (1/3 * (v23 + v13 + v34) + 2/3 * v124) | v1234 < (1/3 * (v24 + v34 + v14) + 2/3 * v123) |
        v1234 < 1/2 * (v12 + v134 + v234) | v1234 < 1/2 * (v13 + v124 + v234) |
        v1234 < 1/2 * (v14 + v123 + v234) | v1234 < 1/2 * (v23 + v134 + v124) |
        v1234 < 1/2 * (v24 + v134 + v123) | v1234 < 1/2 * (v34 + v123 + v124) | v1234 < (v1 + v2 + v3 + v4) | v1234 < (v1 + v234) | v1234 < (v2 + v134) | v1234 < (v3 + v124) | v1234 < (v4 + v123) |
        v1234 < (v1 + v2 + v34) | v1234 < (v1 + v3 + v24) | v1234 < (v1 + v4 + v23) | v1234 < (v2 + v3 + v14) | v1234 < (v2 + v4 + v13) | v1234 < (v3 + v4 + v12) |
        v1234 < (v12 + v34) | v1234 < (v13 + v24) | v1234 < (v14 + v23) |
        v1234 < 1/3 * (v12 + v13 + v14 + v23 + v24 + v34)|
        v1234 < 1/2 * (v23 + v24 + v34) + v1 | v1234 < 1/2 * (v13 + v14 + v34) + v2 | v1234 < 1/2 * (v12 + v14 + v24) + v3 | v1234 < 1/2 * (v12 + v13 + v23) + v4 |
        v1234 < 1/3 * (v123 + v124 + v134 + v234) |
        v1234 < (1/3 * (v12 + v13 + v14) + 2/3 * v234) | v1234 < (1/3 * (v12 + v23 + v24) + 2/3 * v134) |
        v1234 < (1/3 * (v23 + v13 + v34) + 2/3 * v124) | v1234 < (1/3 * (v24 + v34 + v14) + 2/3 * v123) |
        v1234 < 1/2 * (v12 + v134 + v234) | v1234 < 1/2 * (v13 + v124 + v234) |
        v1234 < 1/2 * (v14 + v123 + v234) | v1234 < 1/2 * (v23 + v134 + v124) |
        v1234 < 1/2 * (v24 + v134 + v123) | v1234 < 1/2 * (v34 + v123 + v124) | v1234 < (v2 + v134) | v1234 < (v3 + v124) | v1234 < (v4 + v123) |
        v1234 < (v1 + v2 + v34) | v1234 < (v1 + v3 + v24) | v1234 < (v1 + v4 + v23) | v1234 < (v2 + v3 + v14) | v1234 < (v2 + v4 + v13) | v1234 < (v3 + v4 + v12) |
        v1234 < (v12 + v34) | v1234 < (v13 + v24) | v1234 < (v14 + v23) |
        v1234 < 1/3 * (v12 + v13 + v14 + v23 + v24 + v34) |
        v1234 < 1/2 * (v23 + v24 + v34) + v1 | v1234 < 1/2 * (v13 + v14 + v34) + v2 | v1234 < 1/2 * (v12 + v14 + v24) + v3 | v1234 < 1/2 * (v12 + v13 + v23) + v4 |
        v1234 < 1/3 * (v123 + v124 + v134 + v234) |
        v1234 < (1/3 * (v12 + v13 + v14) + 2/3 * v234)| v1234 < (1/3 * (v12 + v23 + v24) + 2/3 * v134) |
        v1234 < (1/3 * (v23 + v13 + v34) + 2/3 * v124) | v1234 < (1/3 * (v24 + v34 + v14) + 2/3 * v123) |
        v1234 < 1/2 * (v12 + v134 + v234)| v1234 < 1/2 * (v13 + v124 + v234) |
        v1234 < 1/2 * (v14 + v123 + v234) | v1234 < 1/2 * (v23 + v134 + v124) |
        v1234 < 1/2 * (v24 + v134 + v123) | v1234 < 1/2 * (v34 + v123 + v124)) {
      message("The core of 'v' is empty.")
      return(NULL)
    } # Cierre del bloque if
    # Calculamos los vectores de contribuciones marginales
    m <- matrix(nrow = 24, ncol = 4)
    for (i in 1:24) {
      m[i, ] <- marginalvector(v,permutation=i) # aquí cambié lex2bin(v) por v
    }

    if (convexcheck(v)==TRUE){ # aquí cambié lex2bin(v) por v
      vertices <- unique(m, MARGIN = 1)
    } else {
      # Las rectas que definen el núcleo
      # Las desigualdades están en la forma Ax >= b

      # Crear la matriz de coeficientes 'nucleo' y el vector 'b'
      nucleo <- matrix(0, nrow = 14, ncol = 3)  # Matriz de coeficientes A
      b <- numeric(14)  # Vector b
      # Definir las restricciones de la forma Ax >=b
      nucleo[1, ]  <- c(1, 0, 0)
      b[1]  <- v1
      nucleo[2, ]  <- c(-1, 0, 0)
      b[2]  <- -v1234 + v234
      nucleo[3, ]  <- c(0, 1, 0)
      b[3]  <- v2
      nucleo[4, ]  <- c(0, -1, 0)
      b[4]  <- -v1234 + v134
      nucleo[5, ]  <- c(0, 0, 1)
      b[5]  <- v3
      nucleo[6, ]  <- c(0, 0, -1)
      b[6]  <- -v1234 + v124
      nucleo[7, ]  <- c(1, 1, 1)
      b[7]  <- v123
      nucleo[8, ]  <- c(-1, -1, -1)
      b[8]  <- -v1234 + v4
      nucleo[9, ]  <- c(1, 1, 0)
      b[9]  <- v12
      nucleo[10, ] <- c(-1, -1, 0)
      b[10] <- -v1234 + v34
      nucleo[11, ] <- c(1, 0, 1)
      b[11] <- v13
      nucleo[12, ] <- c(-1, 0, -1)
      b[12] <- -v1234 + v24
      nucleo[13, ] <- c(0, 1, 1)
      b[13] <- v23
      nucleo[14, ] <- c(0, -1, -1)
      b[14] <- -v1234 + v14

      # Comprobamos si los vectores de contribuciones marginales están en el core, si están pasan a formar parte

      # PODEMOS QUITAR ESTE CW??
      CW <- matrix(nrow = 24, ncol=4)  # Inicializar matriz vacía con el mismo número de columnas que Wpoints
      checkcore <- numeric(nrow(CW))  # Vector para almacenar los resultados
      for (ii in 1:nrow(m)) {
        checkcore[ii] <- belong2corecheck(v, x=m[ii, 1:4]) # aquí cambié lex2bin(v) por v
      }

      ref <- which(checkcore > 0)  # Encuentra los índices donde checkcore > 0
      # Filtrar filas  donde checkcore==TRUE
      verticesmarginales <- unique(m[ref, , drop = FALSE], MARGIN = 1)

      # Vamos ahora con los que no son vectores de contribuciones marginales

      # Fabricamos matriz con intersecciones
      intersecciones <- matrix(c(
        1, 3, 5, 1, 3, 6, 1, 3, 7, 1, 3, 8, 1, 3, 9, 1, 3, 10, 1, 3, 11, 1, 3, 12,
        1, 3, 13, 1, 3, 14, 1, 4, 5, 1, 4, 6, 1, 4, 7, 1, 4, 8, 1, 4, 9, 1, 4, 10,
        1, 4, 11, 1, 4, 12, 1, 4, 13, 1, 4, 14, 1, 5, 7, 1, 5, 8, 1, 5, 9, 1, 5, 10,
        1, 5, 11, 1, 5, 12, 1, 5, 13, 1, 5, 14, 1, 6, 7, 1, 6, 8, 1, 6, 9, 1, 6, 10,
        1, 6, 11, 1, 6, 12, 1, 6, 13, 1, 6, 14, 1, 7, 9, 1, 7, 10, 1, 7, 11, 1, 7, 12,
        1, 7, 13, 1, 7, 14, 1, 8, 9, 1, 8, 10, 1, 8, 11, 1, 8, 12, 1, 8, 13, 1, 8, 14,
        1, 9, 11, 1, 9, 12, 1, 9, 13, 1, 9, 14, 1, 10, 11, 1, 10, 12, 1, 10, 13, 1, 10,
        14, 1, 11, 13, 1, 11, 14, 1, 12, 13, 1, 12, 14, 2, 3, 5, 2, 3, 6, 2, 3, 7, 2, 3,
        8, 2, 3, 9, 2, 3, 10, 2, 3, 11, 2, 3, 12, 2, 3, 13, 2, 3, 14, 2, 4, 5, 2, 4, 6,
        2, 4, 7, 2, 4, 8, 2, 4, 9, 2, 4, 10, 2, 4, 11, 2, 4, 12, 2, 4, 13, 2, 4, 14, 2,
        5, 7, 2, 5, 8, 2, 5, 9, 2, 5, 10, 2, 5, 11, 2, 5, 12, 2, 5, 13, 2, 5, 14, 2, 6,
        7, 2, 6, 8, 2, 6, 9, 2, 6, 10, 2, 6, 11, 2, 6, 12, 2, 6, 13, 2, 6, 14, 2, 7, 9,
        2, 7, 10, 2, 7, 11, 2, 7, 12, 2, 7, 13, 2, 7, 14, 2, 8, 9, 2, 8, 10, 2, 8, 11,
        2, 8, 12, 2, 8, 13, 2, 8, 14, 2, 9, 11, 2, 9, 12, 2, 9, 13, 2, 9, 14, 2, 10, 11,
        2, 10, 12, 2, 10, 13, 2, 10, 14, 2, 11, 13, 2, 11, 14, 2, 12, 13, 2, 12, 14, 3,
        5, 7, 3, 5, 8, 3, 5, 9, 3, 5, 10, 3, 5, 11, 3, 5, 12, 3, 5, 13, 3, 5, 14, 3, 6,
        7, 3, 6, 8, 3, 6, 9, 3, 6, 10, 3, 6, 11, 3, 6, 12, 3, 6, 13, 3, 6, 14, 3, 7, 9,
        3, 7, 10, 3, 7, 11, 3, 7, 12, 3, 7, 13, 3, 7, 14, 3, 8, 9, 3, 8, 10, 3, 8, 11,
        3, 8, 12, 3, 8, 13, 3, 8, 14, 3, 9, 11, 3, 9, 12, 3, 9, 13, 3, 9, 14, 3, 10, 11,
        3, 10, 12, 3, 10, 13, 3, 10, 14, 3, 11, 13, 3, 11, 14, 3, 12, 13, 3, 12, 14, 4,
        5, 7, 4, 5, 8, 4, 5, 9, 4, 5, 10, 4, 5, 11, 4, 5, 12, 4, 5, 13, 4, 5, 14, 4, 6,
        7, 4, 6, 8, 4, 6, 9, 4, 6, 10, 4, 6, 11, 4, 6, 12, 4, 6, 13, 4, 6, 14, 4, 7, 9,
        4, 7, 10, 4, 7, 11, 4, 7, 12, 4, 7, 13, 4, 7, 14, 4, 8, 9, 4, 8, 10, 4, 8, 11,
        4, 8, 12, 4, 8, 13, 4, 8, 14, 4, 9, 11, 4, 9, 12, 4, 9, 13, 4, 9, 14, 4, 10, 11,
        4, 10, 12, 4, 10, 13, 4, 10, 14, 4, 11, 13, 4, 11, 14, 4, 12, 13, 4, 12, 14, 5,
        7, 9, 5, 7, 10, 5, 7, 11, 5, 7, 12, 5, 7, 13, 5, 7, 14, 5, 8, 9, 5, 8, 10, 5, 8,
        11, 5, 8, 12, 5, 8, 13, 5, 8, 14, 5, 9, 11, 5, 9, 12, 5, 9, 13, 5, 9, 14, 5, 10,
        11, 5, 10, 12, 5, 10, 13, 5, 10, 14, 5, 11, 13, 5, 11, 14, 5, 12, 13, 5, 12, 14,
        6, 7, 9, 6, 7, 10, 6, 7, 11, 6, 7, 12, 6, 7, 13, 6, 7, 14, 6, 8, 9, 6, 8, 10, 6,
        8, 11, 6, 8, 12, 6, 8, 13, 6, 8, 14, 6, 9, 11, 6, 9, 12, 6, 9, 13, 6, 9, 14, 6,
        10, 11, 6, 10, 12, 6, 10, 13, 6, 10, 14, 6, 11, 13, 6, 11, 14, 6, 12, 13, 6, 12,
        14, 7, 9, 11, 7, 9, 12, 7, 9, 13, 7, 9, 14, 7, 10, 11, 7, 10, 12, 7, 10, 13, 7,
        10, 14, 7, 11, 13, 7, 11, 14, 7, 12, 13, 7, 12, 14, 8, 9, 11, 8, 9, 12, 8, 9, 13,
        8, 9, 14, 8, 10, 11, 8, 10, 12, 8, 10, 13, 8, 10, 14, 8, 11, 13, 8, 11, 14, 8,
        12, 13, 8, 12, 14, 9, 11, 13, 9, 11, 14, 9, 12, 13, 9, 12, 14, 10, 11, 13, 10,
        11, 14, 10, 12, 13, 10, 12, 14
      ), ncol = 3, byrow = TRUE)

      CV <- matrix(nrow = 0, ncol = 4)  # Inicializar matriz vacía con 4 columna
      for (II in 1:nrow(intersecciones)) {
        ii <- intersecciones[II, 1]
        jj <- intersecciones[II, 2]
        kk <- intersecciones[II, 3]

        AA <- rbind(nucleo[ii, ], nucleo[jj, ], nucleo[kk, ])  # Los 3 planos del sistema
        BB <- c(b[ii], b[jj], b[kk])  # Coeficientes del sistema
        AABB <- cbind(AA, BB)  # Matriz aumentada del sistema

        if (qr(AA)$rank == 3) {  # Si el sistema es compatible determinado
          hh <- solve(AA, BB)  # Resolver el sistema
          hh[4]=v[15]-sum(hh)
          CV <- rbind(CV, t(hh))  # Agregar la solución como fila a C
        }
        s <- unique(CV, MARGIN = 1)
      }

      checkcore2 <- numeric(nrow(s))
      for (ii in 1:nrow(s)) {
        checkcore2[ii] <- belong2corecheck(v, x=s[ii, 1:4]) # aquí cambié lex2bin(v) por v
      }

      ref2 <- which(checkcore2 > 0)  # Encuentra los índices donde checkcore > 0
      #print(ref2)
      # Filtrar filas de 'm' donde checkcore > 0
      if (length(ref2) > 0) {
        verticesnomarginales <- unique(s[ref2, , drop = FALSE], MARGIN = 1)
      }
      vertices <- unique(rbind(verticesmarginales,verticesnomarginales)) # ahora es matriz, no dataframe
    }
  } # cierre bloque if(n==4)

  return(vertices)

} # Fin de la función



