#' @title Tail game
#' @description Given a sequencing situation without an initial order, this function returns the characteristic function of the associated tail game.
#' @param p A vector containing the processing time of each job.
#' @param alpha A vector containing the cost per unit of time that each job generates while unfinished.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the tail game (interpreted as a cost game), as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given \eqn{S\in 2^N}, \eqn{\Pi(S)} is the set of orders of \eqn{S}, that is, the set of all bijective functions from \eqn{S} to \eqn{\{1,\ldots, s\}}. A generic order of \eqn{S} is denoted by \eqn{\pi_S\in \Pi(S)}.
#' Given \eqn{i\in S} and \eqn{\pi_S\in \Pi(S)}, let \eqn{Pre^{\pi}(i) =\{j\in S : \pi_S(j) <\pi_S(i)\}} be the set of predecessors of \eqn{i} according to order \eqn{\pi_S}.
#'
#' A sequencing situation is a triple \eqn{(N,p,\alpha)} and, possibly, some (information on the) initial order, where \eqn{N=\{1,\ldots,n\}} is a finite set of agents, each one having one job that has to be processed on a machine. To simplify, agent \eqn{i}'s job is identified with \eqn{i}. The processing times
#' of the jobs are given by \eqn{p=(p_i)_{i\in N}} with \eqn{p_i> 0} for all \eqn{i\in N}. For each agent \eqn{i\in N} there is a cost function \eqn{c_i:(0,\infty)\rightarrow \mathbb{R}}, so that \eqn{c_i(t)} represents the cost incurred when job \eqn{i} is completed exactly at time \eqn{t}. Assuming that \eqn{c_i} is linear for all \eqn{i\in N}, there exist \eqn{\alpha_i,\beta_i\geq 0} such that \eqn{c_i(t)=\beta_i + \alpha_i t} for all \eqn{i\in N}, where \eqn{\beta_i} is a fixed service cost and \eqn{\alpha_i t} is the completion cost.
#'
#' For any \eqn{\pi\in \Pi(N)}, \eqn{C(S,\pi)} is the aggregate completion cost of coalition \eqn{S} in the order \eqn{\pi}, formally defined as \deqn{C(S,\pi)=\sum_{i\in S}\alpha_i\Big(p_i+\sum_{j\in Pre^{\pi}(i)}p_j\Big).}
#'
#' A sequencing situation without initial order is a triple \eqn{(N,p,\alpha)} in which there is no information about an initial order.
#'
#' An order that minimizes the aggregate completion cost of coalition \eqn{N} is called an optimal order and denoted by \eqn{\hat{\pi}}. Defining the urgency index of each \eqn{i\in N} as \eqn{u_i=\frac{\alpha_i}{p_i}}, an optimal order can be obtained by arranging jobs in such a way that the corresponding arrangement of their urgency indices is non-increasing. Given a sequencing situation \eqn{(N,p,\alpha)}, \eqn{\Omega(N,p,\alpha)} denotes the set of those optimal orders that also satisfy the following condition: if two jobs share the same urgency index but not the same processing, the one with shortest processing time goes first.
#'
#' The characteristic function of the tail game associated to a sequencing situation \eqn{(N,p,\alpha)}
#' is defined, for each \eqn{S\in 2^N}, by
#'
#' \deqn{c_{tail}(S)=C(S,(\pi_{N\backslash S},\hat{\pi}_S)),}
#'
#' where \eqn{\pi_{N\backslash S}\in\Pi(N\backslash S)} and \eqn{\hat{\pi}_S\in \Omega(S,p_S,\alpha_S)}.
#'
#' Having no information about an initial order, coalitions assume they will be processed at the tail of some "artificial" initial order.
#' @seealso \link{sequencinggame}
#' @examples
#' p <- c(1,2,3,4)
#' alpha <- c(4,5,1,2)
#' tailgame(p,alpha)
#' @references Klijn, F. & Sánchez, E. (2006). Sequencing games without initial order. \emph{Mathematical Methods of Operations Research}, 63, 53-62.
#' @export

tailgame=function(p,alpha,binary=FALSE){ # Principio de la función

  ################################
  ### Comprobación datos entrada##
  ################################

  n=length(p)#Número de trabajos.
  nC=(2^n)-1 #Número de coaliciones.
  if(length(alpha)!=n){
    stop("'p' and 'alpha' must have the same length.")
  }

  if(sum(p<=0)>0){
    stop("All the elements of 'p' must be positive.")
  }

  if(sum(alpha<=0)>0){
    stop("All the elements of 'alpha' must be positive.")
  }

  ################################
  #####Creación de variables######
  ################################

  c=rep(0,nC) #Iniciamos el tail game.
  u=alpha/p #Urgencia.
  optimo=order(u,decreasing=TRUE) #Orden óptimo del trabajo.
  jug=seq(1:n)

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:(nC))
  {
    coalition=getcoalition(S)
    if(length(coalition)==1)
    {
      c[S]=alpha[coalition]*sum(p)#Para las coaliciones individuales, el jugador va el último.
    }else
    {
      Pre=jug[-coalition] #Ponemos primero a los jugadores que no están en la coalición.
      coalition=coalition[order(-u[coalition])]#Ordenamos a los jugadores de S por su urgencia.
      for(j in 1:length(coalition)){#Coste agregado de la coalición S en el orden calculado.
        c[S]=c[S]+alpha[coalition[j]]*(sum(p[coalition[1:j]])+sum(p[Pre]))
      }
    }
  }

  ################################
  ######Salidas de la función#####
  ################################
  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(c))
  } else { # Devolvemos el juego en orden binario.
    return(c)
  }
}# Fin de la función
