#' @title Square upper triangulation
#' @description This function computes a square upper triangular version of the given matrix.
#' @param V A matrix.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return A square upper triangular version of the given matrix.
#' @return This function returns two outputs:
#' \code{SUT}, the square upper triangular matrix.
#' \code{pivot}, a vector indicating pivot rows.
#' @examples
#' set.seed(58)
#' triangularup(matrix(sample(1:10, 16, replace = TRUE), nrow = 4, ncol = 4))
#' triangularup(matrix(c(7,8,5,5,3,5,4,1,3,10,4,4,6,7,8,8),byrow=TRUE, nrow = 4, ncol = 4))
#' triangularup(matrix(c(1,2,1,1,-2,0,1,1),byrow=TRUE, nrow = 2, ncol = 4))
#' triangularup(matrix(c(1,2,1,-2,0,1,3,-1,1,-2,3,3),byrow=TRUE, nrow = 4, ncol = 3))
#' @export

triangularup <- function(V, tol = 100*.Machine$double.eps) {
  # triangularup computes a square upper triangular version of a matrix
  # INPUTS:
  #   V   = A matrix of size mxn.
  #   tol = Tolerance (default 100 * machine epsilon, 100 veces la precisión de la máquina)
  # OUTPUTS:
  #   SUT   = An nxn upper triangular matrix
  #   pivot = Vector indicating pivot rows

  # Si no nos dan la tolerancia, fijar tol = 100 * .Machine$double.eps

  # Ejemplos
  # set.seed(123)
  # V <- matrix(sample(1:10, 16, replace = TRUE), nrow = 4, ncol = 4)
  # triangularup(V)
  # V <- matrix(c(7,8,5,5,3,5,4,1,3,10,4,4,6,7,8,8),byrow=TRUE, nrow = 4, ncol = 4)
  # V <- matrix(c(1,2,1,1,-2,0,1,1),byrow=TRUE, nrow = 2, ncol = 4)
  # V <- matrix(c(1,2,1,-2,0,1,3,-1,1,-2,3,3),byrow=TRUE, nrow = 4, ncol = 3)
  # triangularup(V)
  # Matrix size
  m <- nrow(V)
  n <- ncol(V)

  # Initialize upper triangular matrix
  SUT <- matrix(0, n, n)

  # Initialize pivot row vector
  pivot <- rep(0, m)

  # Loop through the columns of V
  for (jj in 1:n) {
    # Find pivot row, buscamos una fila para pivotar, la primera con elemento no nulo
    ii <- 1
    control <- FALSE # control=0 mientras V(kk,jj)=0

    while (ii <= m && !control) {
      if (abs(V[ii, jj]) > tol) {
        control <- TRUE
      } else {
        ii <- ii + 1 # Si V(ii,jj)=0, paso a la siguiente fila
      }
    }
    #  Ahora, o bien control=0 (No puedo pivotar y entonces paso a la siguiente columna)
    #  o bien control=1 (y hago ceros)

    # If a pivot row is found
    if (control==1) {
      #  Bucle de pivote
      # Primero coloco la fila jj de la matriz triangular SUT
      # que será:  0 0 ... 1 x x x
      # es decir, la fila V(ii,:) de V normalizada desde la posicion jj,
      # Los coeficientes desde el 1 hasta j-1 serán nulos. Como, inicializamos
      # la matriz SUT como la matriz nula, ya no tengo que darles valor.
      # Así pues:
      SUT[jj, jj] <- 1 # Set diagonal element to 1

      # Normalize row in SUT from jj+1 to n
      if (jj < n) {
        SUT[jj, (jj+1):n] <- V[ii, (jj+1):n] / V[ii, jj] # % Normalizo en las columnas jj+1 hasta n
      }

      # Set row elements in V to zero from jj to n
      V[ii, jj:n] <- 0 # Ya no puedo pivotar más veces en la fila ii de la matriz V. Por tanto, pongo 0 en todas las columnas de la jj hasta la última

      # La matriz de pivotes
      pivot[jj] <- ii # Para conseguir un 1 en la columna jj de SUT, pivotamos en la fila ii de V.


      for (fila in (ii+1):m) { # Hago ceros en las demás filas (debajo de la jj)
        if (fila <= m && abs(V[fila, jj]) > tol) { # Si V(fila,jj)\not=0, hago 0 en esta posición
          # en caso contrario, ya hay un 0 y paso a la siguiente fila
          if (jj < n) {
            V[fila, (jj+1):n] <- V[fila, (jj+1):n] - V[fila, jj] * SUT[jj, (jj+1):n]
          }
          V[fila, jj] <- 0
        }
      }
    } # Fin del bucle de pivote
  } # fin del bucle en las columnas de V

  return(list(SUT = SUT, pivot = pivot))
}
