#' @title Unanimity game
#' @description This function returns the characteristic function of the unanimity game of a coalition.
#' @param n Number of players, as an integer.
#' @param S The position of a coalition, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{S} is introduced according to binary order instead of lexicographic order.
#' @return The characteristic function of the unanimity game of coalition \code{S}, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details The characteristic function of the unanimity game of a coalition \eqn{S\in 2^N}
#' is defined, for each \eqn{R\in 2^N}, as \eqn{u_S(R)=1} if \eqn{S\subset R} and \eqn{u_S(R)=0} otherwise.
#' @examples
#' unanimitygame(n=4,S=7)
#' @export
unanimitygame <- function(n, S, binary = FALSE) { # Principio de la función

  ##################################
  ### Comprobación datos entrada ###
  ##################################
  if (n <= 0 | S <= 0) {
    stop("'n' and 'S' must be positive.")
  }

  if (S > ((2^n) - 1)) {
    stop("'S' must not be greater than 2^n-1.")
  }
  if (binary == FALSE) { # Si S se introdujo en lexicográfico, lo pasamos a binario.
    S <- codelex2bin(n, S)
  }

  ################################
  ##### Creación de variables######
  ################################

  nC <- 2^n - 1 # Número de coaliciones
  unanimity <- rep(0, nC) # unanimity game.

  ################################
  ###### Cuerpo de la función######
  ################################

  for (R in S:nC)
  { # S no puede estar contenida en las coaliciones anteriores.
    if (bitwAnd(S, R) == S) { # Si la coalicion en la posicion R es interior a la coalicion en posición S, unanimity(R)=1
      unanimity[R] <- 1
    }
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(unanimity))
  } else { # Devolvemos el juego en orden binario.
    return(unanimity)
  }
} # Fin de la función
