\name{errorevol_ranking_vector_IW}
\alias{errorevol_ranking_vector_IW}

\title{Calculate the error evolution and final predictions of an item-weighted ensemble for rankings}

\description{
  This function calculates the error evolution and final predictions of an item-weigthed ensemble method for ranking data (Albano et al., 2023). }

\usage{
  errorevol_ranking_vector_IW(object, newdata, iw, squared = FALSE)
}

\arguments{
  \item{object}{an object of class 'bagging' or 'boosting' generated by the \code{Ensemble_ranking_IW} function.}

  \item{newdata}{a data frame that can be the same as the one used in the  \code{object} or a new one. Continuous  variables are allowed, while the dummy coding should be used for categorical variables. It must be the output of the \code{prep_data} function.}

  \item{iw}{a weighting vector or matrix. For coherence, \code{iw} should be the same vector/matrix used in \code{Ensemble_ranking_IW(...)}.}

  \item{squared}{logical value indicating whether squared weighting should be used in the final prediction. Default is \code{FALSE}. When set to \code{TRUE}, it corresponds to utilizing the AdaBoost.R.M3 algorithm defined by Albano et al. (2023).}
}

\details{
  This function computes the error and final predictions for a boosting or bagging ranking model using item weighting.}

\value{
  An object of class 'errorevol'. It has two components:
  \item{\code{error}}{ a vector with the error values at each ensemble iteration}
  \item{ \code{final_prediction}}{ a data frame of final predictions for each observation in \code{newdata}.}
}

\references{
  Albano, A., Sciandra, M., and Plaia, A. (2023): "A weighted distance-based approach with boosted decision trees for label ranking." \emph{Expert Systems with Applications}.

  Alfaro, E., Gamez, M., and Garcia, N. (2013): "adabag: An R Package for Classification with Boosting and Bagging." \emph{Journal of Statistical Software}, Vol. 54, 2, pp. 1--35.

  Breiman, L. (1998): "Arcing classifiers." \emph{The Annals of Statistics}, Vol. 26, 3, pp. 801--849.
  
  D'Ambrosio, A.[aut, cre], Amodio, S. [ctb], Mazzeo, G. [ctb], Albano, A. [ctb], Plaia, A. [ctb] (2023). ConsRank: Compute the Median Ranking(s) According to the Kemeny's Axiomatic Approach. R package version 2.1.3, https://cran.r-project.org/package=ConsRank.

  Freund, Y., and Schapire, R.E. (1996): "Experiments with a new boosting algorithm." In \emph{Proceedings of the Thirteenth International Conference on Machine Learning}, pp. 148--156, Morgan Kaufmann.

  Plaia, A., Buscemi, S., Furnkranz, J., and Mencıa, E.L. (2021): "Comparing boosting and bagging for decision trees of rankings." \emph{Journal of Classification}, pages 1–22.

  Zhu, J., Zou, H., Rosset, S., and Hastie, T. (2009): "Multi-class AdaBoost." \emph{Statistics and Its Interface}, 2, pp. 349--360.
}

\examples{
\dontrun{
  # Load simulated ranking data
  data(simulatedRankingData)
  x <- simulatedRankingData$x
  y <- simulatedRankingData$y

  # Prepare the data with item weights
  dati <- prep_data(y, x, iw = c(2, 5, 5, 2))

  # Divide the data into training and test sets
  set.seed(12345)
  samp <- sample(nrow(dati))
  l <- length(dati[, 1])
  sub <- sample(1:l, 2 * l / 3)
  data_sub1 <- dati[sub, ]
  data_test1 <- dati[-sub, ]

  # Apply ensemble ranking with AdaBoost.M1
  boosting_1 <- Ensemble_ranking_IW(
    Label ~ .,
    data = data_sub1,
    iw = c(2, 5, 5, 2),
    mfinal = 3,
    coeflearn = "Breiman",
    control = rpart.control(maxdepth = 4, cp = -1),
    algo = "boosting",
    bin = FALSE
  )

  # Evaluate the performance
  test_boosting1 <- errorevol_ranking_vector_IW(boosting_1, 
    newdata = data_test1, iw=c(2,5,5,2), squared = FALSE)
  test_boosting1.1 <- errorevol_ranking_vector_IW(boosting_1, 
    newdata = data_sub1, iw=c(2,5,5,2), squared = FALSE)

  # Plot the error evolution
  plot.errorevol(test_boosting1, test_boosting1.1)
  }
}



