#' @name Allocation-Methods
#' @export
allocate_prec = function(v0, N, S, lo = NULL, hi = NULL, control = allocation_control())
{
	H = length(N)
	if (is.null(lo)) { lo = rep(1, H) }
	if (is.null(hi)) { hi = rep(Inf, H) }
	stopifnot(H == length(S))
	stopifnot(H == length(lo))
	stopifnot(H == length(hi))

	verbose = control$verbose
	prec_bits = control$bits
	tol = control$tol

	if ("numeric" %in% class(v0)) {
		v0 = mpfr(v0, prec_bits)
	} else if (!("mpfr" %in% class(v0))) {
		stop("v0 must be provided as a single number or mpfr")
	}

	S = mpfr(S, prec_bits)
	hi = pmin(hi, N)
	n = mpfr(lo, prec_bits)

	if (any(n > hi)) {
		stop("There are no feasible solutions")
	}

	r = 0L
	v = sum(N * (N - n) * S^2 / n)

	while (v > v0 && sum(n) < sum(N)) {
		r = r + 1L
		P = N * S / sqrt(n * (n+1)) * (n+1 <= hi)
		h = which.max(asNumeric(P))

		if (verbose) {
			printf("----- About to make selection %d -----\n", r)
			printf("Target v0: %s\n", my_format(v0))
			printf("So far achieved v: %s\n", my_format(v))
			df = data.frame(
				value = my_format(P),
				lo = lo,
				hi = hi,
				n = my_format(n, 0L))
			print(df)
		}

		if (all(P <= tol)) {
			warning("All units from all strata have been selected, but v0 was not attained")
			break
		}

		if (verbose) {
			printf("Now selecting a unit from strata %d\n", h)
		}

		n[h] = n[h] + 1L
		v = sum(N * (N - n) * S^2 / n)
	}

	out = list(n = n, lo = lo, hi = hi, iter = r, v = v, v0 = v0, N = N, S = S)
	structure(out, class = "allocation_prec")
}

#' @export
print.allocation_prec = function(x, control = allocation_control(), ...)
{
	digits = control$digits
	df = data.frame(
		lo = x$lo,
		hi = x$hi,
		n = my_format(x$n, 0L))
	print(df)
	printf("--\n")
	printf("Made %d selections\n", x$reps)
	printf("Target v0: %s\n", my_format(x$v0, digits))
	printf("Achieved v: %s\n", my_format(x$v, digits))
}

#' @export
allocation.allocation_prec = function(object)
{
	asNumeric(object$n)
}
