% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/priors.R
\name{prior}
\alias{prior}
\title{Create prior for Bayesian quantile regression models for complex survey data}
\usage{
prior(
  beta_x_mean = NULL,
  beta_x_cov = NULL,
  sigma_shape = 0.001,
  sigma_rate = 0.001,
  beta_y_mean = NULL,
  beta_y_cov = NULL
)
}
\arguments{
\item{beta_x_mean}{vector of prior means for the regression coefficients. (default = NULL).}

\item{beta_x_cov}{prior covariance matrix for the regression coefficients. (default = NULL).}

\item{sigma_shape}{shape parameter for inverse Gamma prior for \eqn{\sigma^2}. (default = 0.001).}

\item{sigma_rate}{rate parameter for inverse Gamma prior for \eqn{\sigma^2}. (default = 0.001).}

\item{beta_y_mean}{prior means for the coefficients related to the variables that emerge from the product between the orthogonal basis and the outputs
(default = NULL).}

\item{beta_y_cov}{prior covariance matrix for the coefficients related to the variables that emerge from the product between the orthogonal basis and the outputs.
(default = NULL).}
}
\value{
An object of class \code{"prior"}.
}
\description{
\code{prior} creates prior distributions for both single (\code{bqr.svy}) and multiple-output
(\code{mo.bqr.svy}) Bayesian quantile regression models for complex survey data.
}
\details{
The function \code{prior} builds prior distributions for the three methods implemented in the function
\code{bqr.svy} and for the multiple-output quantile regression implemented in the function \code{mo.bqr.svy}.
Every nonspecified prior parameter will get the default value.

\itemize{
\item \code{method = "ald"} in function \code{bqr.svy} allow the specification of hyperparameters
\code{beta_x_mean}, \code{beta_x_cov}, \code{sigma_shape}, and \code{sigma_rate}.
\item \code{method = "score"} in function \code{bqr.svy} allow the specification of hyperparameters
\code{beta_x_mean} and \code{beta_x_cov}.
\item \code{method = "approximate"} in function \code{bqr.svy} allow the specification of hyperparameters
\code{beta_x_mean} and \code{beta_x_cov}.
\item In function \code{mo.bqr.svy}, the specification of hyperparameters \code{beta_x_mean},\code{beta_x_cov},
\code{sigma_shape}, \code{sigma_rate}, \code{beta_y_mean}, and \code{beta_y_cov} are allowed.
}
}
\examples{
\donttest{

# Simulate data
set.seed(123)
n  <- 200
x1 <- rnorm(n, 0, 1)
x2 <- runif(n, -1, 1)
w  <- runif(n, 0.5, 2)   # survey weights

y1 <- 2 + 1.5*x1 - 0.8*x2 + rnorm(n, 0, 1)
y2 <- 1 + 0.5*x1 - 0.2*x2 + rnorm(n, 0, 1)

data <- data.frame(y1 = y1, y2 = y2, x1 = x1, x2 = x2, w = w)

# Define a general informative prior
prior_general <- prior(
  beta_x_mean = c(2, 1.5, -0.8),
  beta_x_cov  = diag(c(0.25, 0.25, 0.25)),
  sigma_shape = 3,
  sigma_rate  = 2,
  beta_y_mean = 1,
  beta_y_cov  = 0.25
)

# Estimate the model parameters with informative prior
fit_ald <- bqr.svy(y1 ~ x1 + x2, weights = w, data = data,
                   prior = prior_general, method = "ald")

fit_scr <- bqr.svy(y1 ~ x1 + x2, weights = w, data = data,
                   prior = prior_general, method = "score")

fit_apx <- bqr.svy(y1 ~ x1 + x2, weights = w, data = data,
                   prior = prior_general, method = "approximate")

# Multiple-output method
fit_mo <- mo.bqr.svy(cbind(y1, y2) ~ x1 + x2, weights = w,
                     data = data, prior = prior_general, n_dir = 10)

plot(fit_ald, type = "trace", which = "x1", tau = 0.5)
plot(fit_ald, type = "trace", which = "x2", tau = 0.5)

print(fit_mo)
}

}
\seealso{
\code{\link{bqr.svy}}, \code{\link{mo.bqr.svy}},
\code{\link{summary}}
}
