
##
# ------------------------------------------------------------------------
#
# "bootglm(model,data,func,B,...)" --
#
# Bootstrap of generalized linear model parameters.
#
# ------------------------------------------------------------------------
##
#' @aliases bootglm
#' @title Bootstrap for Generalized Linear Model
#' @description Parametric bootstrap for generalized linear model.
#' @param model An object of class \code{lm} or \code{glm}.
#' @param data The dataframe used to fit the model.
#' @param B A positive integer; the number of bootstrap replications
#' @param func The function to apply to each sample.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details The parametric bootstrap simply consists in resampling data in the 
#' model with estimated parameters. \code{bootglm} uses this principle 
#' for Generalized Linear Models (GLM) conditionally to the explanatory 
#' variables (see Beran (1997)) for the conditions of validity of this method.
#' 
#' @return \code{bootglm} returns an object of class \code{boodd} 
#' (see \link{class.boodd}).
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek)- submitted.
#' 
#' Beran, R. (1997). Diagnosing Bootstrap Success, \emph{Annals of 
#' the Institute of Statistical Mathematics}, \bold{49}, 1-24.
#' @seealso \code{\link{bootsemi}}.
#' @keywords "Parametric Bootstrap" GLM.
#' @export
#' @examples
#' \donttest{
#' B <- 999
#' x <- runif(100)
#' e <- rnorm(100)
#' y <- x + e>0
#' data <- data.frame(x,y)
#' glm_probit <- glm(y ~ x, family=binomial(link="probit"),data=data) 
#' # Define the function to bootstrap: the mle in the probit model
#' coeff <- function(data){gg=glm(y ~ x, family=binomial(link="probit"),data=data)$coeff}
#' boo1 <- bootglm(glm_probit,data,coeff,B) 
#' # parametric bootstrap of the coeffitients of a probit model
#' plot(boo1,main=c("Boostrap of GLM : probit model","Coefficient"))
#' 
#' # coeffv : a function to return coeff and variance of coefficients 
#' coeffv <- function(data){
#'   gg <- glm(y~ x, family=binomial(link="probit"),data=data) 
#'   var <- diag(summary(gg)$cov.u)
#'   c(gg$coeff,var)
#' } 
#' # Parametric bootstrap of all coeff and variances
#' boo2 <- bootglm(glm_probit,data,coeffv,B)
#' # Construct all type of confidence intervals including bootstrap-t 
#' # and symmetric bootstrap-t 
#' confint(boo2,method="all")
#' 
#' # Family Poisson (Poisson regression)
#' counts <- c(18,17,15,20,10,20,25,13,12)
#' outcome <- gl(3,1,9)
#' treatment <- gl(3,3)
#' data <- data.frame(treatment,outcome,counts)
#' gl <- glm(counts ~ outcome + treatment,family=poisson())
#' meancounts <- function(data) {mean(data$counts)}
#' boo3 <- bootglm(gl,data,meancounts,B)
#' confint(boo3,method="all")
#' plot(boo3)
#' }
  ##
bootglm <- function(model,data,func,B,...) {
  if (class(model)[1] != "glm" & class(model)[1] != "lm") {
    stop("'model' must be an object of class 'glm' or 'lm'")
  }
  
  # Test the value returned by func
  y <- func(data,...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  res <- matrix(nrow=B,ncol=len)
  # Get the name of the response variable
  if (class(model)[1] == "glm") {
    repname <- all.vars(model$formula)[[1]]
  } else {
    repname <- all.vars(model$terms)[[1]]
  }
  
  # Generate B simulations
  sims <- simulate(model,nsim=B)
  for (i in 1:B) {
    data[[repname]] <- sims[[i]]
    res[i,] <- func(data,...)
  }
  
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "bootglm"
  attr(obj,"func") <- func
  return(obj)
}