# ------------------------------------------------------------------------
#
# "confint.boodd <- function(object, parm, level, ...)" --
#
# Calculate a confidence interval for an object of class 'boodd' returned by
# the boots/bootsemi/blockboot/regenboot/etc functions.
#
# ------------------------------------------------------------------------

#' @aliases confint.boodd
#' @title Calculate Confidence Intervals for \code{boodd} Objects.
#' @description 
#' Calculates confidence intervals for an object of class \code{boodd} returned 
#' by bootstrap functions such as \code{\link{boots}}, \code{\link{bootsemi}}, \code{\link{blockboot}}, 
#' \code{\link{regenboot}}, etc.
#' @param object An object of class \code{boodd}.
#' @param parm Not used. Included for consistency with the generic \code{confint()} function.
#' @param level Confidence level. Default is \code{0.95}.
#' @param method Method used to build the confidence interval. 
#'        Choices include:
#'         * \code{perc} - percentile, 
#'         * \code{bperc} - basic percentile, 
#'         * \code{aboot} - asymptotic bootstrap, 
#'         * \code{tboot} - bootstrap-t, 
#'         * \code{tsymboot} - symmetric bootstrap-t, 
#'         *  \code{all} - all the previous methods. 
#'         Default is \code{perc}.
#' @param recenter Logical. If `TRUE` it centers the intervals around
#'        the mean value of the bootstrap samples. Relevant only for 
#'        \code{tboot} or \code{tsymboot} methods. By default it equals `FALSE` 
#'        unless the attribute kind of object `boodd` is `block`.
#' @param ... Optional additional arguments.#' @details 
#' This function provides confidence intervals using several methods, see
#' Efron and Tibshirani (1993), Bertail and Dudek (2025) for description. 
#' The \code{tboot} 
#' and \code{tsymboot} methods require the function to which the bootstrap 
#' method is applied to return an even number of values corresponding to 
#' parameter estimates (first \eqn{k} columns for the parameter of size \eqn{k}) and their variances
#' (columns \eqn{(k+1)} to \eqn{2k}).
#' 
#' @return If the \code{method} argument is not \code{all}, the function
#' \code{confint.boodd} returns a two-column matrix representing the lower and
#' upper bounds of the interval. Each row of the matrix corresponds to the
#' variable to which the interval applies. The default value of the
#' \code{method} argument is \code{perc}. If the \code{method} argument is \code{all},
#' the function \code{confint.boodd} returns a list with the confidence
#' intervals for all supported methods.
#' 
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Efron, B., Tibshirani, R. (1993). \emph{An Introduction to the
#'  Bootstrap}, Chapman and Hall.
#' @seealso \code{\link{plot.boodd}},\code{\link{summary.boodd}}.
#' @keywords "Bootstrap" "Confidence intervals"
#' @examples 
#' B <- 299
#' x <- round(rnorm(15),3)
#' boo1 <- boots(x,mean,B)
#' confint(boo1)
#' confint(boo1,method="bperc")
#' 
#' # bootstrap of several statistics 
#' mv <- function(data) {c(mean(data),var(data)/length(data))} # compute both mean and variance
#' boo2 <- boots(x,mv,B)
#' # Compute both percentile and t-percentile confidence intervals when variance is bootstrapped
#' confint(boo2,method="all")
#' @export
confint.boodd <- function(object, parm = NULL, level = 0.95, 
                          method = c("perc", "bperc", "aboot", "tboot", "tsymboot", "all"),
                          recenter, ...) {
  alpha <- 1 - level
  method <- match.arg(method)
  kind <- attr(object, "kind")
  
  if (missing(recenter)) {
    recenter <- (kind[1] == "block")
  }
  
  s <- object$s
  if (!is.matrix(s)) {
    s <- matrix(s, ncol = 1)
  }
  nc <- ncol(s)
  B <- nrow(s)
  TT <- if (recenter) colMeans(s) else object$Tn
  
  wantAll <- (method == "all")
  if (wantAll) {
    methods <- c("perc", "bperc", "aboot", "tboot", "tsymboot")
    L <- list()
  } else {
    methods <- method
  }
  
  for (meth in methods) {
    if (meth %in% c("tboot", "tsymboot")) {
      if ((nc %% 2) != 0) {
        msg <- paste("The", meth, "method expects an even number of columns.")
        if (wantAll) {
          warning(msg)
          res <- matrix(nrow = 1, ncol = 2)
        } else {
          stop(msg)
        }
      } else {
        k <- nc / 2
        res <- matrix(nrow = k, ncol = 2)
        for (i in 1:k) {
          b <- s[, i]
          V <- s[, i + k]
          if (meth == "tboot") {
            t <- sort((b - TT[i]) / sqrt(V))
            q1 <- valueForPos(t, (B + 1) * alpha / 2)
            q2 <- valueForPos(t, (B + 1) * (1 - alpha / 2))
            res[i, ] <- TT[i] - c(q2, q1) * sqrt(TT[i + k])
          } else {
            t <- sort(abs(b - TT[i]) / sqrt(V))
            q <- valueForPos(t, (B + 1) * (1 - alpha / 2))
            r <- q * sqrt(TT[i + k])
            res[i, ] <- TT[i] + c(-r, r)
          }
        }
        rownames(res) <- colnames(s[, 1:k])
      }
    } else {
      res <- matrix(nrow = nc, ncol = 2)
      for (i in 1:nc) {
        if (meth == "bperc") {
          S <- sort(s[, i])
          lo <- valueForPos(S, (B + 1) * alpha / 2)
          up <- valueForPos(S, (B + 1) * (1 - alpha / 2))
          res[i, ] <- c(lo, up)
        } else if (meth == "perc") {
          S <- sort(s[, i])
          lo <- valueForPos(S, (B + 1) * alpha / 2)
          up <- valueForPos(S, (B + 1) * (1 - alpha / 2))
          res[i, ] <- c(2 * TT[i] - up, 2 * TT[i] - lo)
        } else if (meth == "aboot") {
          sdboot <- sd(s[, i])
          u <- qnorm(1 - alpha / 2)
          res[i, ] <- TT[i] + u * sdboot * c(-1, 1)
        }
      }
    }
    
    colnames(res) <- c("lwr", "upr")
    if (wantAll) {
      L[[meth]] <- res
    }
  }
  
  if (wantAll) return(L) else return(res)
}
