## ------------------------------------------------------------------------
#
# "func_fdb(x, B, kernel = "normal", bandwidth, p = 0.5, PLT = c("spec", "cdf" ,NULL))" --
#
# Function bootstrap in frequency domain (FDB).
#
# ------------------------------------------------------------------------
##'

#' @aliases func_fdb
#' @title Functional Bootstrap in the Frequency Domain (FDB)
#' @description Uses the Frequency Domain Bootstrap (FDB) to compute the bootstrap
#'  spectral density, cumulative distribution function for the estimated spectral 
#'  density, and the quantiles of the standardized distribution.
#' @param x A vector or time series.
#' @param B A positive integer; the number of bootstrap replications.
#' @param kernel A character string specifying the smoothing kernel. The valid values 
#' are:
#' * \code{"normal"} - default,
#' * \code{"epanechnikov"},
#' * \code{"box"} - rectangular kernel.
#' @param bandwidth  A real number; the kernel bandwidth smoothing parameter. 
#' If unspecified, an optimal value is computed using formula \eqn{sd(x)*n^(-1/3)}, 
#' which is smaller than the Silverman's rule-of-thumb bandwidth.
#' @param p A vector of the quantiles to be calculated. Default is \code{p = 0.5}.
#' @param PLT An argument specifying what to plot: 
#' * \code{"spec"} - spectral density,
#' * \code{"cdf"} - cumulative distribution,
#' * \code{NULL} - no plots - default.
#' @details The function performs a bootstrap in the frequency domain and computes 
#' the specified functionals. It estimates the spectral density using
#' a periodogram smoothed by the specified kernel and bandwidth. The function then
#' computes the cumulative spectral density and quantiles based on this estimation.
#' @return The function returns a list of two components:
#' \enumerate{
#'   \item The boodd object:
#'     \itemize{
#'       \item \code{s}: A matrix of size \eqn{B \times (2n_0 + l_p)}, where 
#'       \eqn{n_0 = \lfloor n/2 \rfloor} and \eqn{l_p} is the length of vector \code{p}.
#'       The first \eqn{n_0} columns contain bootstrap values of the spectral density 
#'       at the frequency \eqn{2\pi n_0}. The columns \eqn{n_0+1} up to \eqn{2n_0} contain values
#'       of the cumulative distribution function corresponding to the bootstrap spectral density,
#'       and the last \eqn{l_p} columns contain the estimated quantiles.
#'       \item \code{Tn}: Estimated values of the spectral density, the estimated cumulative 
#'       distribution of the spectral density, and the quantiles.
#'     }
#'   \item \code{freqs}: The vector of frequencies at which the functionals are computed 
#'   for the process \code{x}.
#' }
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P. and  Dudek, A.E. (2021). Consistency of the Frequency Domain
#' Bootstrap for differentiable functionals, \emph{Electron. J. Statist.}, \bold{15}, 1-36.
#' 
#' Hurvich, C. M. and Zeger, S. L. (1987). Frequency domain bootstrap 
#' methods for time series, Technical Report 87-115, Graduate School of Business
#' Administration, New York Univ.
#' 
#' Lahiri, S.N. (2003). \emph{Resampling Methods for Dependent Data}. Springer,
#' New York.
#' @seealso \code{\link{aidedboot}},\code{\link{tft_boot}}, \code{\link{aidedboot}}, 
#' \code{\link{freqboot}}, \code{\link{per_boo}}.
#' @keywords "Frequency Domain Bootstrap" "Spectral analysis" "Time series" 
#' @export
#' @examples
#' \donttest{
#' # Choice of sample size 
#' n <- 1000
#' # Simulate AR(1) model with parameter 0.6
#'  x <- arima.sim(list(order=c(1,0,0),ar=0.6),n=n)
#'  n0=floor(n/2)
#'  bf=func_fdb(x,199, PLT="spec")
#'  }

  
func_fdb<-function (x,B, kernel = "normal", bandwidth,p=0.5,PLT=c("spec","cdf",NULL)) 
{
    if (missing(PLT)) 
      {PLT=NULL}
    PLT=match.arg(PLT)

    n <- length(x)
       if (missing(bandwidth)) {
        bandwidth <- sd(x) * n^(-1/3)
    }
    x <- ts(x, frequency = 1)
    # Compute spectrogram (periodogram)
    P <- spec.pgram(x, plot = FALSE, taper = 0, fast = FALSE, 
        detrend = FALSE)
    specs <- P$spec/(2 * pi)
    freqs <- P$freq * 2 * pi
    n0 <- length(specs)
    # compute renormalized cumulative spectral measure
    cdf=cumsum(P$spec)
    cdf=cdf/cdf[n0]
    # Compute quantile of order p of the renormalized spectral measure
    quant=qd(freqs,cdf,p)

    # Compute smoothed version of spectral density
    # smoothing the periodogram with weights smc
    smc <- smoothingCoefficients(n, bandwidth, kernel)

    P0 <- sum(x)^2/(2 * pi * n)
    In <- c(rev(specs), P0, specs)
    ljn <- c(-rev(freqs), 0, freqs)
    fnhat <- numeric(n0)
    # Compute estimator of spectral density
    for (i in 1:n0) {
        K <- smc[(n0 - i + 1):(3 * n0 - i + 1)]
        fnhat[i] <- 2 * pi * mean(K * In)/bandwidth
    }
    if(!is.null(PLT)) {
    if (PLT=="spec") { 
      plot(freqs,fnhat,type="l")
      }
    else if (PLT=="cdf") {
      plot(freqs,cdf,type="l")
      }
    }

    len=2*n0+length(p)
    n1=n0+1
    n2=2*n0
    n3=2*n0+1
    res <- matrix(nrow = B, ncol = len)

    # compute standardized residuals to be bootstrapped
    eps_hat <- specs/fnhat
    eps_tilde <- eps_hat/mean(eps_hat)
    for (j in 1:B) {
        fnhat_star<-numeric(n0)

        eps_star <- sample(eps_tilde, n0, replace = TRUE)
        I_star <- fnhat * eps_star
    
    P0_star=sample(eps_tilde, 1, replace = TRUE)*P0
    In_star <- c(rev(I_star), P0, I_star)
    for (i in 1:n0) {
        K <- smc[(n0 - i + 1):(3 * n0 - i + 1)]
        fnhat_star[i] <- 2 * pi * mean(K * In_star)/bandwidth
    } 
    res[j,1:n0]=fnhat_star
    # compute renormalized bootstrap spectral density 
    cdf_star=cumsum(I_star)
    cdf_star=cdf_star/cdf_star[n0]
    
    if(!is.null(PLT)) {
    if (PLT=="spec") { lines(freqs,fnhat_star,col=j,type="l") }
    else if (PLT=="cdf") { lines(freqs,cdf_star,col=j,type="l") }
   
    # Re Print original estimator with a line with bigger size 
    if (PLT=="spec") { lines(freqs,fnhat,type="l",lwd=3)}
    else if (PLT=="cdf") { lines(freqs,cdf,type="l",lwd=3) }
    }

    # Compute bootstrap quantile of order p of the renormalized spectral measure
    quant_star=qd(freqs,cdf_star,p)
    
    res[j,n1:n2]=cdf_star
    res[j,n3:len]=quant_star
    }

    obj <- list(s = res, Tn = c(fnhat,cdf,quant))
    class(obj) <- "boodd"
    attr(obj, "kind") <- "frequency"
    attr(obj, "func") <- qd
    return(list(obj,freqs))
}
