# ------------------------------------------------------------------------
# 
# "jackVarField(arr,func,length.block,...)" --
# 
# Jackknife estimator of the variance of function 'func' for random fields.
# 
# ------------------------------------------------------------------------
#' @aliases jackVarField
#' @title Jackknife Variance for Random Fields Based on Blocks
#' @description Estimates the variance of a statistic applied to random fields
#'  using non-overlapping multidimensional blocks.
#' @param arr 
#' A multidimensional real-valued array; it represents a random field on a grid of
#' dimension equal to dimension of the \code{arr}.
#' @param func The function applied to each bootstrap sample.
#' @param length.block An integer or vector of integers; it specified the block lengths for blocks. If a
#' scalar is provided, the same block length is used for all dimensions.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details The \code{\link{jackVarField}} function computes the jackknife variance 
#' estimator for random fields. It involves dividing the array into non-overlapping blocks 
#' of size specified by \code{length.block} and applying the function \code{func},
#'  having output size equal to \emph{p}, to 
#' the data with each block removed in turn. Finally, an empirical
#' variance of the obtained values is computed.
#'  This method is particularly useful for data with spatial or multidimensional
#'   structure.
#' @return Returns a scalar or a covariance matrix, depending on whether the function \code{func}
#'         is univariate or multivariate. For a function returning a vector of length \emph{p},
#'          the output will be a covariance matrix of size \emph{p x p}.
#' @references 
#' Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Carlstein, E. (1986). The use of subseries methods for estimating the variance 
#' of a general statistic from a stationary time series. \emph{Annals of Statist.},
#' \bold{14}, 1171-1179.
#' 
#' Gray, H., Schucany, W. and Watkins, T. (1972). 
#' \emph{The Generalized Jackknife Statistics}. 
#' Marcel Dekker, New York.
#' 
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' @seealso 
#' \code{\link{jackVar}}, 
#' \code{\link{jackFunc}},
#' \code{\link{blockboot}},
#' \code{\link{jackVarBlock}}, 
#' \code{\link{jackFuncRegen}}.
#' 
#' @keywords 
#' Jackknife Resampling "Statistical analysis" "Multidimensional Blocks"
#' @export
#' @examples
#' arr <- array(rnorm(1000), dim = c(10, 10, 10))
#' func <- function(x) { mean(x) }
#' length.block <- c(2, 2, 2)
#' result <- jackVarField(arr, func, length.block) 
##
jackVarField <- function(arr,func,length.block,...) {
  if (any(is.nan(arr))) {
    stop("array contains NaNs")
  }
  # Test the value returned by func
  Tn <- func(arr,...)
  if (!is.vector(Tn)) {
    stop("Function 'func' must return a vector")
  }
  p <- length(Tn)
  Tnames <- names(Tn)	
  dlens <- dim(arr)
  n <- prod(dlens)
  ndims <- length(dlens)
  if (ndims < 2) {
    stop("expected at least 2-dimensional array")
  }
  if (length(length.block) == 1) {
    length.block = rep(length.block,ndims)
  }
  if (length(length.block) != ndims) {
    stop("wrong number of block lengths")
  }
  if (any(length.block > dlens)) {
    stop("block lengths must be less than data lengths in all dimensions")
  }
  if (any(length.block <= 0)) {
    stop("block lengths must be positive")
  }
  blens <- length.block
  bsize <- prod(blens)
  
  # Build the tile indices as strings defining the covering
  # of the array (exceeding values are trimmed)
  tiles <- blockTiles(blens,dlens,canExceed=FALSE)
  tnum <- length(tiles)
  V <- as.vector(arr)
  
  J <- matrix(nrow=tnum,ncol=p)
  for (i in 1:tnum) {
    narr <- arr
    # Indices of the current tile as string
    tile <- tiles[i]
    # Replace values to remove by NaN
    cmd <- paste0("narr[",tile,"] <- NaN")
    eval(parse(text=cmd))
    # Indices of the items to remove		
    bidx <- which(is.nan(narr))
    Tni <- func(V[-bidx],...)
    J[i,] <- (n*Tn - (n-bsize)*Tni)/bsize
  }
  muJ <- colMeans(J)
  V <- matrix(0,nrow=p,ncol=p)
  for (i in 1:tnum) {
    L <- J[i,]-muJ
    V <- V + L%*%t(L)
  }
  V <- bsize*V/(n*tnum)
  if (!is.null(Tnames)) {
    rownames(V) <- Tnames
    colnames(V) <- Tnames
  }
  if (p == 1) {V <- as.vector(V)}
  return(V)
}