# ===========================================================================
# File: "utils.R"
#                        Created: 2020-10-20 12:18:43
#              Last modification: 2020-11-19 10:25:08
# Author: Bernard Desgraupes
# e-mail: <bernard.desgraupes@parisnanterre.fr>
# This file is part of the boodd project.
# ===========================================================================


## 
 # ------------------------------------------------------------------------
 # 
 # "valueForPos(x,pos)" --
 # 
 # If pos is in [n,n+1[, calculate the linear interpolation between x[n] and
 # x[n+1]. The interpolated value is:
 #     x[n]+(x[n+1]-x[n])*(pos-n)
 # 
 # ------------------------------------------------------------------------
 ##
valueForPos <- function(x,pos) {
	len <- length(x)
	if (pos <= 1) {
		n <- 1
	} else if (pos >= len) {
		n <- len-1
	} else {
		n <- floor(pos)
	}
	l <- x[n]
	r <- x[n+1]
	res <- l+(r-l)*(pos-n)
	return(res)
}



## 
 # ------------------------------------------------------------------------
 # 
 # "smallEnsemble(x,s=median(x),eps,delta)" --
 # 
 # Build manually an object of class 'smallEnsemble'. See also the function
 # findBestEpsilon which also computes the same kind of objects.
 # 
 # ------------------------------------------------------------------------
 ##
smallEnsemble <- function(s,eps,delta,trans) {
	res <- list(s=s,epsilon=eps,delta=delta,trans=trans)
	class(res) <- "smallEnsemble"
    return(res)
}



# # See also in package stats
# # The Nadaraya–Watson kernel regression estimate.
# # ksmooth(x, y, kernel = c("box", "normal"), bandwidth = 0.5,
# #         range.x = range(x),
# #         n.points = max(100L, length(x)), x.points)





## 
 # ------------------------------------------------------------------------
 # 
 # "smoothingCoefficients <- function(n0,h,kernel)" --
 # 
 # Return the quantities K(\lambda_{jn}/h) for j= -n,...,+n. The returned
 # vector has length 2*(n%/%2)+1 and is symmetrical. The ((n%/%2)+1)-th value
 # corresponds to frequency 0.
 # 
 # ------------------------------------------------------------------------
 ##
smoothingCoefficients <- function(n,bandwidth,kernel="normal") {
  h=bandwidth
	N <- 2*(n%/%2)
	# Adjust the kernel name
	kernel <- match.arg(kernel,c("normal","box","epanechnikov"))
	kernel <- substr(kernel,1,3)
	# Compute the coefficients
	if (kernel == "nor") {
		Q <- 2*pi*(0:N)/(h*n)
		K <- dnorm(Q)
	} else {
		K <- numeric(N+1)
		jm <- floor(h*n/(2*pi))
		idx <- 0:jm
		Q <- 2*pi*idx/(h*n)
		if (kernel == "epa") {
			# Indices are 1-based
			K[idx+1] <- 0.75*(1-Q^2)
		} else {
			K[idx+1] <- rep(0.5,length(Q))
		}
	}
	# Complete by symmetry
	K <- c(rev(K[-1]),K)
	return(K)
}


 # ------------------------------------------------------------------------
 # 
 # "blockString(origs,blens)" --
 # 
 # Build a string representing a block extracted from an array. 'origs' is
 # the vector of start indices on all the dimensions. 'blens' is the vector
 # of block lengths.
 # 
 # Example:
 #     > blockString(c(4,9,5),c(2,3,5))
 #     [1] "4:5,9:11,5:9"
 # 
 # ------------------------------------------------------------------------
 ##
blockString <- function(origs,blens) {
	ndims <- length(blens)

	substr <- character(0)
	for (i in 1:ndims) {
		substr <- c(substr,paste0(as.character(origs[i]),":",as.character(origs[i]+blens[i]-1)))
	}
	str <- paste0(substr,collapse=",")
	return(str)
}


## 
 # ------------------------------------------------------------------------
 # 
 # "blockTiles(blens,dlens,canExceed)" --
 # 
 # Build all the tile indices covering the array. 'blens' are the block
 # lengths and 'dlens' the data lengths. Return the indices as a vector of
 # strings. The 'canExceed' argument has the same meaning as with the
 # segmentStrings function.
 # 
 # Example:
 #     > blockTiles(c(3,4,5),c(9,10,10))
 #     "1:3,1:4,1:5"   "4:6,1:4,1:5"   "7:9,1:4,1:5"  
 #     "1:3,5:8,1:5"   "4:6,5:8,1:5"   "7:9,5:8,1:5"  
 #     "1:3,9:12,1:5"  "4:6,9:12,1:5"  "7:9,9:12,1:5" 
 #     "1:3,1:4,6:10"  "4:6,1:4,6:10"  "7:9,1:4,6:10" 
 #     "1:3,5:8,6:10"  "4:6,5:8,6:10"  "7:9,5:8,6:10" 
 #     "1:3,9:12,6:10" "4:6,9:12,6:10" "7:9,9:12,6:10"
 #     > blockTiles(c(3,4,5),c(9,10,10),FALSE)
 #     "1:3,1:4,1:5"  "4:6,1:4,1:5"  "7:9,1:4,1:5" 
 #     "1:3,5:8,1:5"  "4:6,5:8,1:5"  "7:9,5:8,1:5" 
 #     "1:3,1:4,6:10" "4:6,1:4,6:10" "7:9,1:4,6:10"
 #     "1:3,5:8,6:10" "4:6,5:8,6:10" "7:9,5:8,6:10"
 # 
 # ------------------------------------------------------------------------
 ##
blockTiles <- function(blens,dlens,canExceed=TRUE) {
	bl <- length(blens)
	dl <- length(dlens)
	if (bl < 2) {
		stop("expected at least 2 dimensions")
	}
	if (bl != dl) {
		stop("'blens' and 'dlens' must have same length")
	}
	out <- outer(segmentStrings(blens[1],dlens[1],canExceed),segmentStrings(blens[2],dlens[2],canExceed),FUN="paste",sep=",")
	if (bl > 2) {
		for (i in 3:bl) {
			out <- outer(out,segmentStrings(blens[i],dlens[i],canExceed),FUN="paste",sep=",")
		}
	}
	return(as.vector(out))
}


## 
 # ------------------------------------------------------------------------
 # 
 # "segmentStrings(bl,dl,canExceed)" --
 # 
 # Build a vector of strings corresponding to the successive intervals along a 
 # direction. 'bl' is the block length and 'dl' the data length. The last 
 # range may exceed the data length (it will be trimmed in the fieldboot 
 # function) unless the 'canExceed' argument is set to FALSE.
 # 
 # Example:
 #     > segmentStrings(3,10)
 #     [1] "1:3"   "4:6"   "7:9"   "10:12"
 #     > segmentStrings(3,10,canExceed=FALSE)
 #     [1] "1:3"   "4:6"   "7:9"
 # 
 # ------------------------------------------------------------------------
 ##
segmentStrings <- function(bl,dl,canExceed=TRUE) {
	nb <- (dl-1)%/%bl+1
	if (!canExceed & dl%%bl > 0) {
		nb <- nb-1
	}
	str <- character(0)
	for (i in 1:nb) {
		str <- c(str,paste0(as.character(bl*(i-1)+1),":",as.character(bl*i)))
	}
	return(str)
}


## 
 # ------------------------------------------------------------------------
 # 
 # "completeArray <- function(arr,blens)" --
 # 
 # Extend an array to handle circular blocks in random fields bootstrap. 
 # The function returns an array in which all the dimensions have been 
 # completed in order to be a multiple of the corresponding block length.
 # 
 # For instance, if 'arr' has dim 2x3x4 and if the block lengths are 2,2,3 
 # then the returned array has dim 2x4x6.
 # 
 # ------------------------------------------------------------------------
 ##
completeArray <- function(arr,blens) {
	ndim <- length(blens)
	perm <- seq(along=blens)
	perm <- c(perm[2:ndim],perm[1])
	perm <- c(2:ndim,1)

	for (i in 1:ndim) {
		arr <- completeLastDim(arr,blens[ndim])
		blens <- c(blens[2:ndim],blens[1])
		arr <- aperm(arr,perm)
	}
	return(arr)
}


## 
 # ------------------------------------------------------------------------
 # 
 # "completeLastDim(arr,bl)" --
 # 
 # Complete the last dimension of an array by circularity to make it a
 # multiple of 'bl' (block length). 
 # 
 # ------------------------------------------------------------------------
 ##
completeLastDim <- function(arr,bl) {
	dims <- dim(arr)
	ndim <- length(dims)
# 	dl <- dims[ndim]
# 	compl <- bl-dl%%bl
	compl <- bl-1
	len <- prod(dims[-ndim])
	v <- as.vector(arr)
	v <- c(v,v[1:(compl*len)])
	newdims <- c(dims[-ndim],dims[ndim]+compl)
	narr <- array(v,dim=newdims)
	return(narr)
}


## 
# ------------------------------------------------------------------------
# 
# "f_hat(x,X_1)" 
# 
# Estimator of the marginal distribution density of the tarnsition kernel
#
# ------------------------------------------------------------------------
##
f_hat=function(x,X1){
  n=length(X1)
  h=n^{-1/5}
  mean(dnorm((x-X1)/h))/h 
}

## 
# ------------------------------------------------------------------------
# 
# "f2_hat(x,y,X_1)" 
# 
# Compute the density of the transition Kernel starting from x
#
# ------------------------------------------------------------------------
##
f2_hat <- function(x,y,X) {
  n <- length(X)-1
  Xi <- X[1:n]
  Xip1 <- X[2:(n+1)]
  h=n^(-1/5)
  f <- numeric(n)
  for (i in 1:n) {
    f<- mean(dnorm((x-Xi)/h)*dnorm((y-Xip1)/h))/mean(dnorm((x-X)/h))/h
  }
  return(f)
}

## 
# ------------------------------------------------------------------------
# 
# "qd(f,cdf,p)" 
# 
# Compute the quantile of order p for 
#
# ------------------------------------------------------------------------
##
qd=function(f, cdf, p){
  lp=length(p)
  result_ind=rep(0,lp)
  for (i in 1:lp){
    result_ind[i]=which(cdf==min(cdf[cdf>p[i]]))
    result_ind[i]=f[result_ind[i]]
  }
  return(result_ind)
}

## 
# ------------------------------------------------------------------------
# 
# "sdens_est(per_norm,kernel,n,h)" 
# 
# Spectral Density Estimation from Normalized Periodogram
#
# ------------------------------------------------------------------------
##
sdens_est<-function(per_norm,kernel,n,h){
  #per_norm is the normalised periodogram such that its expectation is equal to the spectral density
  prob<-get_prob(n,h,kernel)
  N<-n/2-1  #n even
  wind<-trunc(n*h)
  prob2<-c(prob[(wind+1):2],prob)
  per<-c(per_norm[(N-wind+2):N],0,per_norm,0,per_norm[1:(wind-1)])
  ## attention: middle term set equal to 0
  sdens<-rep(0,times=N)
  
  for(j in 1:N){
    sdens[j]<-weighted.mean(per[j:(j+2*wind)],prob2)
  }
  return(sdens)
}

## 
# ------------------------------------------------------------------------
# 
# "get_prob(n,h,kernel)" 
# 
# Probability Weights for Spectral Density Estimation
#
# ------------------------------------------------------------------------
##

get_prob<-function(n,h,kernel){
  #kernel=0: daniell; else: bartlett-priestley (epanechnikov)
  wind<-trunc(n*h)
  prob<-rep(0,times=wind+1)
  if(kernel==0){
    prob<-rep(1/(2*wind+1),times=wind+1)
  }
  else{
    for(j in 1:(wind+1)){
      prob[j]<-1-((j-1)/wind)^2
    }
    prob<-prob/(1+2*sum(prob[2:(wind+1)]))
  }
  return(prob)
}



## 
# ------------------------------------------------------------------------
# 
# "qgh(q,g,h)" 
# 
# g-and-h distribution : random generation, quantile , cdf and density 
#
# ------------------------------------------------------------------------
##


qgh <- function(q,g,h) {
  Zp <- qnorm(q)
  ## not vectorized!
  if (g==0) Zp else (exp(g*Zp)-1)*exp((h*Zp^2/2))/g
}

## since the quantile function is defined, it makes generating
##  random values easy!
rgh <- function(n,g,h) {
  qgh(runif(n),g,h)
}


## 
# ------------------------------------------------------------------------
# 
# "pgh(q,g,h, eps=1e-7)" 
# 
# Cdf of g-h distribution
#
# ------------------------------------------------------------------------
##


pgh <- function(p,g,h,eps=1e-7) {
  uniroot(function(z) { qgh(z,g,h) - p}, interval=c(eps,1-eps))$root
}

## 
# ------------------------------------------------------------------------
# 
# "dgh(q,g,h, log=FALSE, ndep=1e-3,...)" 
# 
# Density function of g-h distribution
#
# ------------------------------------------------------------------------
##

dgh <- function(x,g,h,log=FALSE,ndep=1e-3,...) {
  ## crude vectorization in x (not g or h)
  if (length(x)>1) return(sapply(x,dgh,g=g,h=h,log=log,ndep=ndep,...))
  r <- (pgh(x+ndep,g,h)-pgh(x,g,h))/ndep
  if (log) log(r) else r
}


## 
# ------------------------------------------------------------------------
# 
# "is.bad(x)" 
# 
# Function to test for NA or INF values
#
# ------------------------------------------------------------------------
##

is.bad <- function(x) any(is.na(x) | is.infinite(x))






