% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regenboot.R
\name{regenboot}
\alias{regenboot}
\title{Regenerative and Approximative Regenerative Block Bootstrap.}
\usage{
regenboot(
  x,
  func,
  B,
  ...,
  atom,
  small = NULL,
  s = median(x),
  plotIt = FALSE,
  moon = length(x)
)
}
\arguments{
\item{x}{A numeric vector representing a Markov process.}

\item{func}{The function to apply to each sample.}

\item{B}{A positive integer; the number of bootstrap replications.}

\item{...}{Optional additional arguments for the \code{func} function.}

\item{atom}{A numeric value or a string; an atom of the Markov chain in the atomic case.}

\item{small}{An object of class \code{\link{smallEnsemble}}.
It can be created optimally using the function \code{\link{findBestEpsilon}}.}

\item{s}{A real number specifying the center of the small set. Default is the median of \code{x}.}

\item{plotIt}{Logical. If \code{TRUE} then the function returns a plot of the time
series with the approximative regenerative blocks. Does not plot anything in
the atomic case.
Default is \code{FALSE}.}

\item{moon}{A positive integer. Default is length of \code{x}.
\code{moon} should be smaller than the length of \code{x}, then it creates bootstrap
samples of size \code{moon}.}
}
\value{
returns an object of class \code{boodd}.
}
\description{
Performs regenerative block bootstrap and approximately regenerative
block bootstrap on a Markov chain, either in the atomic case
or in the general Harris case.
}
\details{
This function \code{regenboot} implements two different kinds of
regenerative bootstrap:
\itemize{
\item \emph{A regenerative block bootstrap} used for atomic Markov
chains.
\item \emph{An approximate regenerative block bootstrap} used to bootstrap
Harris recurrent Markov chains
based on a given small set of the form \eqn{[s-eps,s+eps]} where \emph{s}
is the center and \emph{eps} the radius.
}

One must specify either the \code{atom} argument or the \code{small}
argument. In the first case, \code{atom} is the state used to split the
Markov chain into blocks ending with the atom. In the second case,
\code{small} is an object of class \code{smallEnsemble} representing the
small ensemble. Such objects are typically
obtained using the \code{findBestEpsilon} function but may also be constructed
manually using the \code{smallEnsemble} function.
}
\examples{
\donttest{
B <- 299
n <- 200

# Atomic Boostrap
acgt <- c("A","C","G","T")
probs <- c(.3,.1,.3,.3)
atom <- "C"
set.seed(1)
x <- sample(acgt,n,prob=probs,repl=TRUE)
propAtom <- function(x) {
  tbl <- as.vector(table(x))
  prop <- tbl[3]/length(x)
   return(prop)
}
boo <- regenboot(x,propAtom,B,atom=atom)
plot(boo)

# Approximate regenerative bootstrap with estimated small set
ar <- arima.sim(list(c(1,0,0),ar=0.6),n=500)
# Find the small ensemble with the largest number of regenerations
sm <- findBestEpsilon(ar,s=0,plotIt=TRUE)
# Approximate regenerative bootstrap of the mean
rboo <- regenboot(ar,mean,small=sm,B=999, plotIt=TRUE)
# Plot the corresponding bootstrap distribution
plot(rboo)
# Compute the bootstrap percentile confidence interval
confint(rboo)
}
}
\references{
Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for
Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.

Bertail, P., Clémençon, S. (2006a). Regenerative Block
Bootstrap for Markov Chains. \emph{Bernoulli},  \bold{12}, 689-712.

Bertail, P. and Clémençon, S. (2006b).
Regeneration-based statistics for Harris recurrent
Markov chains.
\emph{Lecture notes in Statistics}, vol. \bold{187}, pp. 1-54, Springer.

Radulović, D. (2004). Renewal type bootstrap for Markov chains. \emph{Test},
\bold{13}, 147-192.
}
\seealso{
\code{\link{boots}},
\code{\link{blockboot}},
\code{\link{plot.boodd}},
\code{\link{confint.boodd}}, \code{\link{findBestEpsilon}},\code{\link{smallEnsemble}}.
}
\keyword{"Approximative}
\keyword{"Atom"}
\keyword{"Markov}
\keyword{"Regenerative}
\keyword{"Small}
\keyword{Block}
\keyword{Bootstrap"}
\keyword{Regenerative}
\keyword{chain"}
\keyword{set"}
