#'Generate images with bounding boxes from coordinates in a dataframe
#'
#'\code{outBox} adds colored bounding boxes and labels to images based on
#'predetermined coordinates, classifiers, and image dimensions captured in a
#'dataframe.
#'
#'Load a dataframe that captures data points for each image to process.
#'The file format is the same as generated by boxer and boxer2.
#'
#'Note y-coordinates should extend downward from the top, not upward from the
#'bottom.
#'
#'@param points_master A dataframe that captures the name of the image file,
#'the four coordinates of the bounding box, the width and height dimensions
#'of the image, the classifier, and the color for the label and box. The file
#'format is a follows:
#'file_name : (chr) name of the image file;
#'x_left    : (int) left side x-coordinate;
#'y_top     : (int) top side y-coordinate;
#'x_right   : (int) right side x-coordinate;
#'y_bottom  : (int) bottom side y-coordinate;
#'size_x    : (int) image width in pixels;
#'size_y    : (int) image height in pixels;
#'classifier: (chr) classifier used as label;
#'color     : (chr) name of color to use for box and label.
#'@param file_path_input The directory that contains the image files.
#'@param file_path_output The directory where to save annotated images.
#'@param show_classifier Logical to include the classifier above the bounding.
#'box. The default is F.
#'
#'@return A new image file with the bounding box annotation for each of the
#'  images processed and resized accordingly. The name of each output image
#'  file will be the same as the corresponding input file prefixed with "out_".
#'
#' @examples
#'points_master <- data.frame(read.csv(system.file("extdata","points_example.csv",
#'package = "boundingbox")), stringsAsFactors = FALSE)
#'
#'
#'outBox(points_master = points_master,
#'file_path_input = system.file("extdata", package = "boundingbox"),
#'file_path_output = tempdir(), show_classifier = TRUE)
#'
#'@export
outBox <- function(points_master, file_path_input, file_path_output, show_classifier = FALSE){

    # run checks to validate parameters
    checks <- c(12:15, 2:4)
    paramCheck(names = points_master$file_name, file_path_input = file_path_input, color = NA,
               outbox = TRUE, file_path_output = file_path_output, checks = checks,
               points_master = points_master)

    for (fname in points_master$file_name){

       image <- recaptureImage(file_path_input = file_path_input, fname = fname,
                               size_x = points_master[points_master$file_name == fname,"size_x"][1],
                               size_y = points_master[points_master$file_name == fname,"size_y"][1])


       points_boxed <- points_master[points_master$file_name == fname,]



    #DRAW THE BOUNDING BOX OVER THE IMAGE AND SAVE AS OUTPUT
    if (tools::file_ext(fname) =="png"){
        grDevices::png(filename= file.path(file_path_output, paste0("out_",fname)),
                       width =  points_master[points_master$file_name == fname,"size_x"][1],
                       height = points_master[points_master$file_name == fname,"size_y"][1])

    } else if (tools::file_ext(fname) =="bmp") {
        grDevices::bmp(filename= file.path(file_path_output, paste0("out_",fname)),
                       width =  points_master[points_master$file_name == fname,"size_x"][1],
                       height = points_master[points_master$file_name == fname,"size_y"][1])

    } else {
        grDevices::jpeg(filename= file.path(file_path_output, paste0("out_",fname)),
                        width =  points_master[points_master$file_name == fname,"size_x"][1],
                        height = points_master[points_master$file_name == fname,"size_y"][1],
                        quality = 100)
    }

    graphics::plot(image)

    for (n in 1:nrow(points_boxed)){

        lab <- ifelse(show_classifier == TRUE, points_boxed$classifier[n], "")
        color <- gplots::col2hex(points_boxed[n, "color"])

        graphics::lines(c(points_boxed$x_left[n], points_boxed$x_left[n]),
                        c(points_boxed$size_y[n] - points_boxed$y_top[n], points_boxed$size_y[n] - points_boxed$y_bottom[n]),
                        lwd = (points_boxed$size_x[n]/360),  col = color)
        graphics::lines(c(points_boxed$x_right[n], points_boxed$x_right[n]),
                        c(points_boxed$size_y[n] - points_boxed$y_top[n], points_boxed$size_y[n] - points_boxed$y_bottom[n]),
                        lwd = (points_boxed$size_x[n]/360),  col = color)
        graphics::lines(c(points_boxed$x_left[n], points_boxed$x_right[n]),
                        c(points_boxed$size_y[n] - points_boxed$y_top[n], points_boxed$size_y[n] - points_boxed$y_top[n]),
                        lwd = (points_boxed$size_x[n]/360),  col = color)
        graphics::lines(c(points_boxed$x_left[n], points_boxed$x_right[n]),
                        c(points_boxed$size_y[n] - points_boxed$y_bottom[n], points_boxed$size_y[n] - points_boxed$y_bottom[n]),
                        lwd = (points_boxed$size_x[n]/360),  col = color)
        graphics::text(x = (points_boxed$x_left[n] + points_boxed$x_right[n])/2, y = (points_boxed$size_y[n] - points_boxed$y_top[n] + 5),
                       label = lab, col = color, cex = (points_boxed$size_x[n]/360))
    }

    grDevices::dev.off()


    }
    message(paste0("Files created in ", file_path_output))
}



