#' Compute bootstrapping results
#'
#' Wrapper for computing \code{n_bt} bootstrap replica, combining the
#' functionality of \code{compute_intermediate_results} and
#' \code{summarise_intermediate_results}.
#'
#' @param base_compare A data.frame as generated by \code{create_comparison}.
#' @param n_bt An integer number of resamples to be used for bootstrapping.
#' @param grouping_var A character vector of variables that must be present in
#'   \code{base_compare}.
#' @param seed A seed passed to resampling step for reproducibility.
#' @param ps_flags A list as returned by \code{set_ps_flags}.
#' @param label_distribution Expects a data.frame with columns \code{"label_id",
#'   "label_freq", "n_docs"}. \code{label_freq} corresponds to the number of
#'   occurences a label has in the gold standard. \code{n_docs} corresponds to
#'   the total number of documents in the gold standard.
#' @param cost_fp A numeric value > 0, defaults to NULL.
#' @inheritParams option_params
#'
#' @return A data.frame containing \code{n_bt} boot replica of results as
#'   returned by \code{compute_intermediate_results} and
#'   \code{summarise_intermediate_results}.
#'
generate_replicate_results <- function(
    base_compare,
    n_bt,
    grouping_var,
    seed = NULL,
    ps_flags = list("intermed" = FALSE, "summarise" = FALSE),
    label_distribution = NULL,
    cost_fp = NULL,
    replace_zero_division_with = options::opt("replace_zero_division_with"),
    drop_empty_groups = options::opt("drop_empty_groups"),
    progress = options::opt("progress")) {
  stopifnot(is.data.frame(base_compare))
  stopifnot(is.integer(n_bt))
  stopifnot("gold" %in% colnames(base_compare))
  stopifnot(is.logical(progress))

  doc_id_list <- dplyr::distinct(
    dplyr::filter(base_compare, .data$gold == TRUE),
    .data$doc_id
  )

  # core resampling is done by rsample library
  if (!is.null(seed)) {
    set.seed(seed)
  }
  boot <- suppressWarnings(
    rsample::bootstraps(data = doc_id_list, times = n_bt, apparent = TRUE)
  )
  boot_dfs <- stats::setNames(boot$splits, boot$id)
  boot_dfs <- purrr::map(boot_dfs, as.data.frame)

  # apply wrapper to each of the bootstrap replica
  # note: a call to furrr attaches purrr
  boot_results <- furrr::future_map_dfr(
    boot_dfs,
    .f = helper_f,
    .progress = progress,
    .options = furrr::furrr_options(seed = seed),
    .id = "boot_replicate",
    compare_cpy = base_compare,
    ps_flags = ps_flags,
    label_distribution = label_distribution,
    cost_fp = cost_fp,
    grouping_var = grouping_var,
    replace_zero_division_with = replace_zero_division_with,
    drop_empty_groups = drop_empty_groups
  )

  boot_results
}

#' Calculate bootstrapping results for one sample
#'
#' Internal wrapper for computing bootstrapping results on one sample, combining
#' the functionality of \code{compute_intermediate_results} and
#' \code{summarise_intermediate_results}.
#'
#' @param sampled_id_list A list of all doc_ids of this bootstrap.
#' @param compare_cpy As created by \code{create_comparison}.
#' @param grouping_var A vector of variables to be used for aggregation.
#' @param ps_flags A list as returned by \code{set_ps_flags}.
#' @param label_distribution Expects a data.frame with columns \code{"label_id",
#'   "label_freq", "n_docs"}. \code{label_freq} corresponds to the number of
#'   occurences a label has in the gold standard. \code{n_docs} corresponds to
#'   the total number of documents in the gold standard.
#' @param cost_fp A numeric value > 0, defaults to NULL.
#' @inheritParams option_params
#'
#' @return A data.frame as returned by \code{summarise_intermediate_results}.
helper_f <- function(
    sampled_id_list,
    compare_cpy,
    grouping_var,
    label_distribution = NULL,
    ps_flags = list("intermed" = FALSE, "summarise" = FALSE),
    cost_fp = NULL,
    replace_zero_division_with = options::opt("replace_zero_division_with"),
    drop_empty_groups = options::opt("drop_empty_groups")) {
  compare_resampled <- collapse::join(
    compare_cpy, sampled_id_list,
    on = "doc_id", how = "inner", verbose = 0,
    multiple = TRUE,
    validate = "m:m"
  )

  intermediate_results_resampled <- compute_intermediate_results(
    compare_resampled,
    grouping_var,
    propensity_scored = ps_flags$intermed,
    cost_fp = cost_fp,
    drop_empty_groups = drop_empty_groups
  )
  summarise_intermediate_results(
    intermediate_results_resampled,
    propensity_scored = ps_flags$summarise,
    label_distribution = label_distribution,
    replace_zero_division_with = replace_zero_division_with,
    set = TRUE
  )
}

#' @describeIn generate_replicate_results Variant with dplyr based
#' internals rather than collapse internals.
generate_replicate_results_dplyr <- function( # nolint styler: off
    base_compare,
    n_bt,
    grouping_var,
    seed = NULL,
    label_distribution = NULL,
    ps_flags = list("intermed" = FALSE, "summarise" = FALSE),
    cost_fp = NULL,
    progress = FALSE) {
  stopifnot(is.data.frame(base_compare))
  stopifnot(is.integer(n_bt))
  stopifnot("gold" %in% colnames(base_compare))
  stopifnot(is.logical(progress))

  doc_id_list <- dplyr::distinct(
    dplyr::filter(base_compare, .data$gold == TRUE),
    .data$doc_id
  )

  # core resampling is done by rsample library
  if (!is.null(seed)) {
    set.seed(seed)
  }
  boot <- suppressWarnings(
    rsample::bootstraps(data = doc_id_list, times = n_bt, apparent = TRUE)
  )
  boot_dfs <- stats::setNames(boot$splits, boot$id)
  boot_dfs <- purrr::map(boot_dfs, as.data.frame)

  # apply wrapper to each of the bootstrap replica
  # note: a call to furrr attaches purrr
  boot_results <- furrr::future_map_dfr(
    boot_dfs,
    .f = helper_f_dplyr,
    .progress = progress,
    .id = "boot_replicate",
    .options = furrr::furrr_options(seed = seed),
    compare_cpy = base_compare,
    ps_flags = ps_flags,
    label_distribution = label_distribution,
    cost_fp = cost_fp,
    grouping_var = grouping_var
  )

  boot_results
}

#' Calculate bootstrapping results for one sample
#'
#' Internal wrapper for computing bootstrapping results on one sample, combining
#' the functionality of \code{compute_intermediate_results} and
#' \code{summarise_intermediate_results}.
#'
#' @param sampled_id_list A list of all doc_ids of this bootstrap.
#' @param compare_cpy As created by \code{create_comparison}.
#' @param grouping_var A vector of variables to be used for aggregation.
#' @param ps_flags A list with logicals \code{"intermed"} and
#'   \code{"summarise"}.
#' @param label_distribution Expects a data.frame with columns \code{"label_id",
#'   "label_freq", "n_docs"}. \code{label_freq} corresponds to the number of
#'   occurences a label has in the gold standard. \code{n_docs} corresponds to
#'   the total number of documents in the gold standard.
#' @param cost_fp A numeric value > 0, defaults to NULL.
#'
#' @return A data.frame as returned by
#'   \code{summarise_intermediate_results_dplyr}.
helper_f_dplyr <- function(
    sampled_id_list,
    compare_cpy,
    grouping_var,
    ps_flags = list("intermed" = FALSE, "summarise" = FALSE),
    label_distribution = NULL,
    cost_fp = NULL) {
  compare_resampled <- dplyr::inner_join(
    compare_cpy, sampled_id_list,
    by = "doc_id",
    relationship = "many-to-many"
  )
  intermediate_results_resampled <- compute_intermediate_results_dplyr(
    compare_resampled,
    rlang::syms(grouping_var),
    propensity_scored = ps_flags$intermed,
    cost_fp
  )
  summarise_intermediate_results_dplyr(
    intermediate_results_resampled,
    propensity_scored = ps_flags$summarise,
    label_distribution = label_distribution
  )
}
