% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{remove_cardinal_biases}
\alias{remove_cardinal_biases}
\title{Remove cardinal biases}
\usage{
remove_cardinal_biases(
  err,
  x,
  space = "180",
  bias_type = "fit",
  plots = "hide",
  poly_deg = 4,
  var_sigma = TRUE,
  var_sigma_poly_deg = 4,
  reassign_at_boundaries = TRUE,
  reassign_range = 2,
  break_points = NULL,
  init_outliers = NULL,
  debug = FALSE,
  do_plots = NULL
)
}
\arguments{
\item{err}{a vector of errors, deviations of response from the true stimuli}

\item{x}{a vector of true stimuli in degrees (see space)}

\item{space}{circular space to use (a string: \code{180} or \code{360})}

\item{bias_type}{bias type to use (\code{fit}, \code{card}, \code{obl}, or \code{custom}, see details)}

\item{plots}{a string \code{hide}, \code{show}, or \code{return} to hide, show, or return plots (default: \code{hide})}

\item{poly_deg}{degree of the fitted polynomials for each bin (default: 4)}

\item{var_sigma}{allow standard deviation (width) of the fitted response distribution to vary as a function of distance to the nearest cardinal (default: True)}

\item{var_sigma_poly_deg}{degree of the fitted polynomials for each bin for the first approximation for the response distribution to select the best fitting model (default: 4)}

\item{reassign_at_boundaries}{select the bin for the observations at the boundaries between bins based on the best-fitting polynomial (default: True)}

\item{reassign_range}{maximum distance to the boundary at which reassignment can occur (default: 2 degrees)}

\item{break_points}{can be used to assign custom break points instead of cardinal/oblique ones with \code{bias_type} set to \code{custom} (default: NULL)}

\item{init_outliers}{a vector determining which errors are initially assumed to be outliers (default: NULL)}

\item{debug}{print some extra info (default: False)}

\item{do_plots}{deprecated, use the parameter \code{plots} instead}
}
\value{
If \code{plots=='return'}, returns the three plots showing the biases
(combined together with \code{\link[patchwork:wrap_plots]{patchwork::wrap_plots()}}). Otherwise, returns a list with the following elements:
\itemize{
\item is_outlier - 0 for outliers (defined as \verb{±3*pred_sigma} for the model with varying sigma or as \verb{±3\\*SD} for the simple model)
\item pred predicted error
\item be_c error corrected for biases (\verb{be_c = observed error - pred})
\item which_bin the numeric ID of the bin that the stimulus belong to
\item bias the bias computed as described above
\item bias_typ bias type (cardinal or oblique)
\item pred_lin predicted error for a simple linear model for comparison
\item pred_sigma predicted SD of the error distribution
\item coef_sigma_int, coef_sigma_slope intercept and slope for the sigma prediction

}
}
\description{
Remove cardinal biases
}
\details{
If the \code{bias_type} is set to \code{fit}, the function computes the cardinal biases in the following way:
\enumerate{
\item Create two sets of bins, splitting the stimuli vector into bins centered at cardinal and at oblique directions.
\item For each set of bins, fit a nth-degree polynomial for the responses in each bin, optionally allowing the distribution of responses to vary in width as a function of distance to the nearest cardinal (regardless of whether the bins are centered at the cardinal or at the oblique, the width of the response distribution usually increases as the distance to cardinals increase).
\item Choose the best-fitting model between the one using cardinal and the one using oblique bins.
\item Compute the residuals of the best-fitting model - that's your bias-corrected error - and the biases (see below).
}
The bias is computed by flipping the sign of errors when the average predicted error is negative, so, that, for example, if on average the responses are shifted clockwise relative to the true values, the trial-by-trial error would count as bias when it is also shifted clockwise.

If \code{bias_type} is set to \code{obl} or \code{card}, only one set of bins is used, centred at cardinal or oblique angles, respectively.
}
\examples{

# Data in orientation domain from Pascucci et al. (2019, PLOS Bio),
# https://doi.org/10.5281/zenodo.2544946

ex_data <- Pascucci_et_al_2019_data[observer == 4, ]
remove_cardinal_biases(ex_data$err, ex_data$orientation, plots = "show")

# Data in motion domain from Bae & Luck (2018, Neuroimage),
# https://osf.io/2h6w9/
ex_data_bae <- Bae_Luck_2018_data[subject_Num == unique(subject_Num)[5], ]
remove_cardinal_biases(ex_data_bae$err, ex_data_bae$TargetDirection,
  space = "360", plots = "show"
)

# Using a stricter initial outlier boundary

remove_cardinal_biases(ex_data_bae$err, ex_data_bae$TargetDirection,
  space = "360", plots = "show",
  init_outliers = abs(ex_data_bae$err) > 60
)

# We can also use just one bin by setting `bias_type` to custom
# and setting the `break_points` at the ends of the range for x

remove_cardinal_biases(ex_data_bae$err, ex_data_bae$TargetDirection,
  space = "360", bias_type = "custom",
  break_points = c(-180, 180), plots = "show",
  reassign_at_boundaries = FALSE, poly_deg = 8,
  init_outliers = abs(ex_data_bae$err) > 60
)

}
\references{
{
\itemize{
\item Chetverikov, A., & Jehee, J. F. M. (2023). Motion direction is represented as a bimodal probability distribution in the human visual cortex. Nature Communications, 14(7634). \doi{10.1038/s41467-023-43251-w}
\item van Bergen, R. S., Ma, W. J., Pratte, M. S., & Jehee, J. F. M. (2015). Sensory uncertainty decoded from visual cortex predicts behavior. Nature Neuroscience, 18(12), 1728–1730. \doi{10.1038/nn.4150}
}
}
}
