% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulCoef.R, R/confint.cirls.R, R/vcov.cirls.R
\name{simulCoef}
\alias{simulCoef}
\alias{confint.cirls}
\alias{vcov.cirls}
\title{Simulate coefficients, calculate Confidence Intervals and Variance-Covariance Matrix for a \code{cirls} object.}
\usage{
simulCoef(object, nsim = 1, seed = NULL, complete = TRUE,
  constrained = TRUE)

\method{confint}{cirls}(object, parm, level = 0.95, nsim = 1000,
  complete = TRUE, ...)

\method{vcov}{cirls}(object, complete = TRUE, nsim = 1000, trunc = TRUE,
  ...)
}
\arguments{
\item{object}{A fitted \code{cirls} object.}

\item{nsim}{The number of simulations to perform.}

\item{seed}{Either NULL or an integer that will be used in a call to \code{\link[=set.seed]{set.seed()}} before simulating the coefficients.}

\item{complete}{If FALSE, doesn't return inference for undetermined coefficients in case of an over-determined model.}

\item{constrained}{A logical switch indicating Whether to simulate from the constrained (the default) or unconstrained coefficients distribution.}

\item{parm}{A specification of which parameters to compute the confidence intervals for. Either a vector of numbers or a vector of names. If missing, all parameters are considered.}

\item{level}{The confidence level required.}

\item{...}{Further arguments passed to or from other methods. For \code{vcov} and \code{confint} can be used to provide a \code{seed} for the internal coefficient simulation.}

\item{trunc}{If set to \code{FALSE} the unmodified GLM variance-covariance computed within \code{\link[=summary.glm]{summary.glm()}} is returned.}
}
\value{
For \code{simulCoef}, a matrix with \code{nsim} rows containing simulated coefficients.

For \code{confint}, a two-column matrix with columns giving lower and upper confidence limits for each parameter.

For \code{vcov}, a matrix of the estimated covariances between the parameter estimates of the model.
}
\description{
Simulates coefficients for a fitted \code{cirls} object. \code{confint} and \code{vcov} compute confidence intervals and the Variance-Covariance matrix for coefficients from a fitted \code{cirls} object. These methods supersede the default methods for \code{cirls} objects.
}
\details{
\code{confint} and \code{vcov} are custom methods for \link[=cirls.fit]{cirls} objects to supersede the default methods used for \link[stats:glm]{glm} objects. Internally, they both call \code{simulCoef} to generate coefficient vectors from a Truncated Multivariate Normal Distribution using the \code{\link[TruncatedNormal:rtmvnorm]{TruncatedNormal::rtmvnorm()}} function. This distribution accounts for truncation by constraints, ensuring all coefficients are feasible with respect to the constraint matrix. \code{simulCoef} typically doesn't need to be used directly for confidence intervals and variance-covariance matrices, but it can be used to check other summaries of the coefficients distribution.

These methods only work when \code{Cmat} is of full row rank, i.e. if there are less constraints than variables in \code{object}.
}
\note{
By default, the Variance-Covariance matrix generated by \code{vcov} is different than the one returned by \code{summary(obj)$cov.scaled}. The former accounts for the reduction in degrees of freedom resulting from the constraints, while the latter is the unconstrained GLM Variance-Covariance. Note that the unconstrained one can be obtained from \code{vcov} by setting \code{constrained = FALSE}.
}
\examples{
####################################################
# Isotonic regression

#----- Perform isotonic regression

# Generate data
set.seed(222)
p1 <- 5; p2 <- 3
x1 <- matrix(rnorm(100 * p1), 100, p1)
x2 <- matrix(rnorm(100 * p2), 100, p2)
b1 <- runif(p1) |> sort()
b2 <- runif(p2)
y <- x1 \%*\% b1 + x2 \%*\% b2 + rnorm(100, sd = 2)

# Fit model
Ciso <- diff(diag(p1))
resiso <- glm(y ~ x1 + x2, method = cirls.fit, Cmat = list(x1 = Ciso))

#----- Extract uncertainty

# Extract variance covariance
vcov(resiso)

# Extract confidence intervals
confint(resiso)

# We can extract the usual unconstrained matrix
vcov(resiso, constrained = FALSE)
all.equal(vcov(resiso, constrained = FALSE), summary(resiso)$cov.scaled)

# Simulate from the distribution of coefficients
sims <- simulCoef(resiso, nsim = 10)

# Check that all simulated coefficient vectors are feasible
apply(resiso$Cmat \%*\% t(sims) >= resiso$lb, 2, all)
}
\references{
Geweke, J.F., 1996. Bayesian Inference for Linear Models Subject to Linear Inequality Constraints, in: Lee, J.C., Johnson, W.O., Zellner, A. (Eds.), Modelling and Prediction Honoring Seymour Geisser. \emph{Springer, New York, NY}, pp. 248–263. \doi{10.1007/978-1-4612-2414-3_15}

Botev, Z.I., 2017, The normal law under linear restrictions: simulation and estimation via minimax tilting, \emph{Journal of the Royal Statistical Society, Series B}, \strong{79} (\strong{1}), pp. 1–24.
}
\seealso{
\link[TruncatedNormal:tmvnorm]{rtmvnorm} for the underlying routine to simulate from a TMVN. \code{\link[=checkCmat]{checkCmat()}} to check if the contraint matrix can be reduced.
}
