% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meanvar.R
\name{cpss.mean}
\alias{cpss.mean}
\title{Detecting changes in mean}
\usage{
cpss.mean(
  dataset,
  algorithm = "BS",
  dist_min = floor(log(n)),
  ncps_max = ceiling(n^0.4),
  pelt_pen_val = NULL,
  pelt_K = 0,
  wbs_nintervals = 500,
  criterion = "CV",
  times = 2,
  Sigma = NULL
)
}
\arguments{
\item{dataset}{a numeric matrix of dimension \eqn{n\times d}, where each row represents an observation and each column stands for a variable. A numeric vector is also acceptable for univariate observations.}

\item{algorithm}{a character string specifying the change-point searching algorithm, one of the following choices: "SN" (segment neighborhood), "BS" (binary segmentation), "WBS" (wild binary segmentation) and "PELT" (pruned exact linear time) algorithms.}

\item{dist_min}{an integer specifying minimum searching distance (length of feasible segments).}

\item{ncps_max}{an integer specifying an upper bound of the number of true change-points.}

\item{pelt_pen_val}{a numeric vector specifying candidate values of the penalty only if \code{algorithm = "PELT"}.}

\item{pelt_K}{a numeric value for pruning adjustment only if \code{algorithm = "PELT"}. It is usually taken to be 0 if the negative log-likelihood is used as a cost, see Killick et al. (2012).}

\item{wbs_nintervals}{an integer specifying the number of random intervals drawn only if \code{algorithm = "WBS"}, see Fryzlewicz (2014).}

\item{criterion}{a character string specifying the model selection criterion, "CV" ("cross-validation") or "MS" ("multiple-splitting").}

\item{times}{an integer specifying how many times of sample-splitting should be performed; It should be 2 if \code{criterion = "CV"}.}

\item{Sigma}{if a numeric matrix (or constant) is supplied, it will be taken as the value of the common covariance (or variance). By default it is \code{NULL}, and the covariance is estimated by \deqn{\widehat{\Sigma} = \frac{1}{2(n-1)}\sum_{i=1}^{n-1} (Y_i-Y_{i+1})(Y_i-Y_{i+1})';}}
}
\value{
\code{cpss.mean} returns an object of an \proglang{S4} class, called "\code{cpss}", which collects data and information required for further change-point analyses and summaries. See \code{\link{cpss.custom}}.
}
\description{
Detecting changes in mean
}
\examples{
library("cpss")
set.seed(666)
n <- 2048
tau <- c(205, 267, 308, 472, 512, 820, 902, 1332, 1557, 1598, 1659)
seg_len <- diff(c(0, tau, n))
mu <- rep(c(0, 14.64, -3.66, 7.32, -7.32, 10.98, -4.39, 3.29, 19.03, 7.68, 15.37, 0), seg_len)
ep <- 7 * rnorm(n)
y <- mu + ep
\donttest{
res <- cpss.mean(y, algorithm = "SN", ncps_max = 20)
summary(res)
# 205  267  307  471  512  820  897  1332  1557  1601  1659
plot(res, type = "scatter")
plot(res, type = "path")
out <- update(res, dim_update = 12)
out@cps
# 205  267  307  471  512  820  897 1332 1557 1601 1659 1769
# coef(out)
}
}
\references{
Killick, R., Fearnhead, P., and Eckley, I. A. (2012). Optimal Detection of Changepoints With a Linear Computational Cost. Journal of the American Statistical Association, 107(500): 1590–1598.
Fryzlewicz, P. (2014). Wild binary segmentation for multiple change-point detection. The Annals of Statistics, 42(6): 2243–2281.
}
\seealso{
\code{\link{cpss.meanvar}} \code{\link{cpss.var}}
}
