% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visualize_distances.R
\name{visualize_distances}
\alias{visualize_distances}
\title{Visualize Distance Matrices via MDS, Heatmap, or Network Graph}
\usage{
visualize_distances(
  dist_mat,
  method = c("mds_classic", "mds_weighted", "heatmap", "qgraph"),
  k = 3,
  weights = NULL,
  group = NULL,
  main_title = NULL,
  tol = 1e-10,
  ...
)
}
\arguments{
\item{dist_mat}{A square distance matrix (numeric matrix) or a \code{dist} object.}

\item{method}{Character string specifying the visualization method. Options are:
\itemize{
  \item \code{"mds_classic"}: Classical MDS (cmdscale).
  \item \code{"mds_weighted"}: Weighted MDS (wcmdscale, requires \code{weights}).
  \item \code{"heatmap"}: Heatmap with optional clustering and group annotations.
  \item \code{"qgraph"}: Network graph representation of similarity.
}}

\item{k}{Integer. Number of dimensions to retain for MDS (default 3). Must be \code{>=1} and \code{<= min(4, n_obs-1)}.}

\item{weights}{Optional numeric vector of weights for weighted MDS. Must match the number of observations.}

\item{group}{Optional factor or vector indicating group membership for coloring plots.}

\item{main_title}{Optional character string specifying the main title of the plot.}

\item{tol}{Numeric tolerance for checking approximate symmetry (default 1e-10).}

\item{...}{Additional arguments passed to internal plotting functions (\code{plot_heatmap} or \code{plot_qgraph}).}
}
\value{
The plotting object is returned and automatically printed:
\itemize{
  \item MDS plots return a \code{ggmatrix} from \code{GGally}.
  \item Heatmaps return a \code{pheatmap} object.
  \item Network graphs are plotted directly (returns \code{NULL}).
}
}
\description{
This function provides a unified interface to visualize distance matrices
using classical or weighted Multidimensional Scaling (MDS),
heatmaps, or network graphs. Group annotations can be provided for coloring.
}
\details{
\code{visualize_distances} is a wrapper around three internal plotting functions:
\itemize{
  \item \code{plot_mds}: Creates a pairwise scatterplot matrix of MDS coordinates with density plots on the diagonal.
  \item \code{plot_heatmap}: Plots a heatmap of the distance matrix with hierarchical clustering and optional group annotations.
  \item \code{plot_qgraph}: Plots a network graph where nodes represent observations and edges represent similarity.
}

The function validates that \code{dist_mat} is square, symmetric, and has zero diagonal elements.
If a distance matrix has a \code{trimmed_idx} attribute and \code{group} is not provided,
a factor indicating "Trimmed" vs "Outlier" is created automatically.
}
\examples{
# Load iris dataset
data(iris)

# Compute Euclidean distances on numeric columns
dist_iris <- dist(iris[, 1:4])

# Create a grouping factor based on Species
group_species <- iris$Species

# --------------------------------------
# Classical MDS (2D)
# --------------------------------------
visualize_distances(
  dist_mat = dist_iris,
  method = "mds_classic",
  k = 2,
  group = group_species,
  main_title = "Classical MDS - Iris Dataset - Euclidean Distance"
)

# --------------------------------------
# Weighted MDS (uniform weights)
# --------------------------------------
weights <- rep(1, nrow(iris))
visualize_distances(
  dist_mat = dist_iris,
  method = "mds_weighted",
  k = 2,
  weights = weights,
  group = group_species,
  main_title = "Weighted MDS - Iris Dataset - Euclidean Distance"
)

# --------------------------------------
# Heatmap (limit rows to 30)
# --------------------------------------
visualize_distances(
  dist_mat = dist_iris,
  method = "heatmap",
  group = group_species,
  main_title = "Iris Heatmap by Species - Euclidean Distance",
  max_n = 30,
  palette = "YlGnBu",
  clustering_method = "complete",
  annotation_legend = TRUE,
  stratified_sampling = TRUE,
  seed = 123
)

# --------------------------------------
# Network Graph (limit nodes to 30)
# --------------------------------------
visualize_distances(
  dist_mat = dist_iris,
  method = "qgraph",
  group = group_species,
  max_nodes = 30,
  label_size = 2,
  edge_threshold = 0.1,
  layout = "spring",
  seed = 123,
  main_title = "Iris Network Graph by Species - Euclidean Distance"
)

}
\seealso{
\code{\link[stats]{cmdscale}} for classical MDS.
\code{\link[vegan]{wcmdscale}} for weighted MDS.
\code{\link[pheatmap]{pheatmap}} for heatmaps.
\code{\link[qgraph]{qgraph}} for network graphs.
\code{\link[GGally]{ggpairs}} for MDS scatterplot matrices.
}
