#' Compute Coefficients for Sparse Density-Convoluted SVM
#'
#' Computes the coefficients or returns the indices of nonzero coefficients at chosen values of \code{lambda} from a fitted \code{\link{dcsvm}} object.
#'
#' @name coef.dcsvm
#' @aliases coef.dcsvm
#' @title Compute Coefficients for Sparse Density-Convoluted SVM
#'
#' @description
#' Computes the coefficients or indices of nonzero coefficients at specified \code{lambda} values from a fitted \code{dcsvm} model.
#'
#' @usage
#' \method{coef}{dcsvm}(object, s = NULL, type = c("coefficients", "nonzero"), ...)
#'
#' @param object A fitted \code{\link{dcsvm}} object.
#' @param s Value(s) of the L1 tuning parameter \code{lambda} for computing coefficients. Default is the entire \code{lambda} sequence obtained by \code{\link{dcsvm}}.
#' @param type \code{"coefficients"} or \code{"nonzero"}? \code{"coefficients"} computes the coefficients at given values for \code{s}; \code{"nonzero"} returns a list of the indices of the nonzero coefficients for each value of \code{s}. Default is \code{"coefficients"}.
#' @param ... Not used. Other arguments to \code{predict}.
#'
#' @details
#' \code{s} is the vector of \code{lambda} values at which predictions are requested. If \code{s} is not in the lambda sequence used for fitting the model, the \code{coef} function uses linear interpolation. The new values are interpolated using a fraction of coefficients from both left and right \code{lambda} indices.
#'
#' @return
#' Either the coefficients at the requested values of \code{lambda}, or a list of the indices of the nonzero coefficients for each \code{lambda}.
#'
#' @seealso
#' \code{\link{predict.dcsvm}}
#'
#' @examples
#' data(colon)
#' fit <- dcsvm(colon$x, colon$y, lam2=1)
#' c1 <- coef(fit, type="coefficients", s=c(0.1, 0.005))
#' c2 <- coef(fit, type="nonzero")
#'
#' @export
coef.dcsvm <- function(object, s=NULL, 
    type=c("coefficients", "nonzero"), ...) {
  type <- match.arg(type)
  b0 <- t(as.matrix(object$b0))
  rownames(b0) <- "(Intercept)"
  nbeta <- rbind2(b0, object$beta)
  if (!is.null(s)) {
    vnames <- dimnames(nbeta)[[1]]
    dimnames(nbeta) <- list(NULL, NULL)
    lambda <- object$lambda
    lamlist <- lambda.interp(lambda, s)
    nbeta <- nbeta[,lamlist$left,drop=FALSE] %*% 
      Diagonal(x=lamlist$frac) +
      nbeta[,lamlist$right,drop=FALSE] %*% 
      Diagonal(x=1-lamlist$frac)
    dimnames(nbeta) <- list(vnames, paste(seq(along=s)))
  }
  if (type == "coefficients") 
    return(nbeta)
  if (type == "nonzero") 
    return(nonzero(nbeta[-1, , drop=FALSE], bystep=TRUE))
} 