% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ALC.R
\name{ALC}
\alias{ALC}
\alias{ALC.gp}
\alias{ALC.dgp2}
\alias{ALC.dgp3}
\title{Active Learning Cohn for Sequential Design}
\usage{
ALC(object, x_new, ref, cores)

\method{ALC}{gp}(object, x_new = NULL, ref = NULL, cores = 1)

\method{ALC}{dgp2}(object, x_new = NULL, ref = NULL, cores = 1)

\method{ALC}{dgp3}(object, x_new = NULL, ref = NULL, cores = 1)
}
\arguments{
\item{object}{object of class \code{gp}, \code{dgp2}, or \code{dgp3}}

\item{x_new}{matrix of possible input locations, if object has been run 
through \code{predict} the previously stored \code{x_new} is used}

\item{ref}{optional reference grid for ALC approximation, if \code{ref = NULL} 
then \code{x_new} is used}

\item{cores}{number of cores to utilize in parallel, by default no 
parallelization is used}
}
\value{
list with elements:
\itemize{
  \item \code{value}: vector of ALC values, indices correspond to \code{x_new}
  \item \code{time}: computation time in seconds
}
}
\description{
Acts on a \code{gp}, \code{dgp2}, or \code{dgp3} object. 
   Current version requires squared exponential covariance 
   (\code{cov = "exp2"}).  Calculates ALC over the input locations 
   \code{x_new} using specified reference grid.  If no reference grid is 
   specified, \code{x_new} is used as the reference.  Optionally utilizes 
   SNOW parallelization.  User should 
   select the point with the highest ALC to add to the design.
}
\details{
Not yet implemented for Vecchia-approximated fits or Matern kernels.

    All iterations in the object are used in the calculation, so samples 
    should be burned-in.  Thinning the samples using \code{trim} will
    speed up computation.  This function may be used in two ways:
    \itemize{
        \item Option 1: called on an object with only MCMC iterations, in 
              which case \code{x_new} must be specified
        \item Option 2: called on an object that has been predicted over, in 
              which case the \code{x_new} from \code{predict} is used
    }
    In Option 2,  it is recommended to set \code{store_latent = TRUE} for 
    \code{dgp2} and \code{dgp3} objects so
    latent mappings do not have to be re-calculated.  Through \code{predict}, 
    the user may specify a mean mapping (\code{mean_map = TRUE}) or a full 
    sample from the MVN distribution over \code{w_new} 
    (\code{mean_map = FALSE}).  When the object has not yet been predicted
    over (Option 1), the mean mapping is used.
    
    SNOW parallelization reduces computation time but requires more memory
    storage.  C code derived from the "laGP" package (Robert B Gramacy and 
    Furong Sun).
}
\examples{
# Additional examples including real-world computer experiments are available at: 
# https://bitbucket.org/gramacylab/deepgp-ex/

# --------------------------------------------------------
# Example 1: toy step function, runs in less than 5 seconds
# --------------------------------------------------------

f <- function(x) {
    if (x <= 0.4) return(-1)
    if (x >= 0.6) return(1)
    if (x > 0.4 & x < 0.6) return(10*(x-0.5))
}

x <- seq(0.05, 0.95, length = 7)
y <- sapply(x, f)
x_new <- seq(0, 1, length = 100)

# Fit model and calculate ALC
fit <- fit_two_layer(x, y, nmcmc = 100, cov = "exp2")
fit <- trim(fit, 50)
fit <- predict(fit, x_new, cores = 1, store_latent = TRUE)
alc <- ALC(fit)

\donttest{
# --------------------------------------------------------
# Example 2: damped sine wave
# --------------------------------------------------------

f <- function(x) {
    exp(-10*x) * (cos(10*pi*x - 1) + sin(10*pi*x - 1)) * 5 - 0.2
}

# Training data
x <- seq(0, 1, length = 30)
y <- f(x) + rnorm(30, 0, 0.05)

# Testing data
xx <- seq(0, 1, length = 100)
yy <- f(xx)

plot(xx, yy, type = "l")
points(x, y, col = 2)

# Conduct MCMC (can replace fit_two_layer with fit_one_layer/fit_three_layer)
fit <- fit_two_layer(x, y, D = 1, nmcmc = 2000, cov = "exp2")
plot(fit)
fit <- trim(fit, 1000, 2)

# Option 1 - calculate ALC from MCMC iterations
alc <- ALC(fit, xx)

# Option 2 - calculate ALC after predictions
fit <- predict(fit, xx, cores = 1, store_latent = TRUE)
alc <- ALC(fit)

# Visualize fit
plot(fit)
par(new = TRUE) # overlay ALC
plot(xx, alc$value, type = 'l', lty = 2, axes = FALSE, xlab = '', ylab = '')

# Select next design point
x_new <- xx[which.max(alc$value)]
}

}
\references{
Sauer, A., Gramacy, R.B., & Higdon, D. (2023). Active learning for deep 
    Gaussian process surrogates. *Technometrics, 65,* 4-18.  arXiv:2012.08015
    \cr\cr
Seo, S, M Wallat, T Graepel, and K Obermayer. 2000. Gaussian Process Regression:
    Active Data Selection and Test Point Rejection. In Mustererkennung 2000, 
    2734. New York, NY: SpringerVerlag.\cr\cr
Gramacy, RB and F Sun. (2016). laGP: Large-Scale Spatial Modeling via Local 
    Approximate Gaussian Processes in R. \emph{Journal of Statistical Software 
    72} (1), 1-46. doi:10.18637/jss.v072.i01
}
