#' Plot of the bootstrap f2 simulation
#'
#' This is a method for the function \code{plot()} for objects of class
#' \sQuote{\code{bootstrap_f2}}.
#'
#' @param x An object of class \sQuote{\code{bootstrap_f2}} returned by the
#'   \code{\link{bootstrap_f2}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[boot]{plot.boot}()} function.
#'
#' @details The element \code{Boot} of the \sQuote{\code{bootstrap_f2}} object
#' that is returned by the function \code{\link{bootstrap_f2}()} is an object
#' of class \sQuote{\code{boot}}, generated by the function
#' \code{\link[boot]{boot}()} from the \sQuote{\code{boot}} package. Thus, the
#' corresponding \code{plot} method is used. Arguments to the
#' \code{\link[boot]{plot.boot}()} function can be passed via the \code{...}
#' parameter. In addition to making the plot the function prints the result of
#' Shah's lower 90\% BCa confidence interval to the console.
#'
#' @return The \sQuote{\code{bootstrap_f2}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{bootstrap_f2}}, \code{\link[boot]{boot}},
#' \code{\link[boot]{plot.boot}}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_plot.bootstrap_f2.R
#'
#' @export

plot.bootstrap_f2 <- function(x, ...) {
  mf <- match.call(expand.dots = TRUE)
  m <- match("digits", names(mf), 0L)

  if (m == 0) {
    digits <- getOption("digits")
  } else {
    digits <- mf[[m]]
  }

  plot(x$Boot, ...)

  cat("\nShah's lower 90% BCa confidence interval:\n",
      formatC(as.numeric(x$Shah_BCa_CI[1]), digits = digits),
      "\n\n")

  invisible(x)
}

#' Summary of the bootstrap f2 simulation
#'
#' This is a method for the function \code{summary()} for objects of class
#' \sQuote{\code{bootstrap_f2}}.
#'
#' @param object An object of class \sQuote{\code{bootstrap_f2}} returned by
#'   the \code{\link{bootstrap_f2}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[boot]{print.boot}()} and
#'   \code{\link[boot]{print.bootci}()} functions.
#'
#' @details The elements \code{Boot} and \code{CI} of the
#' \sQuote{\code{bootstrap_f2}} object that is returned by the function
#' \code{\link{bootstrap_f2}()} are objects of type \sQuote{\code{boot}} and
#' \sQuote{\code{bootci}}, respectively, generated by the functions
#' \code{\link[boot]{boot}()} and \code{\link[boot]{boot.ci}()}, respectively,
#' from the \sQuote{\code{boot}} package. Thus, the corresponding \code{print}
#' methods are used. Arguments to the \code{\link[boot]{print.boot}()} and
#' \code{\link[boot]{print.bootci}()} functions can be passed via the
#' \code{...} parameter.
#'
#' @return The \sQuote{\code{bootstrap_f2}} object passed to the \code{object}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{bootstrap_f2}}, \code{\link[boot]{boot}},
#' \code{\link[boot]{boot.ci}}, \code{\link[boot]{print.boot}},
#' \code{\link[boot]{print.bootci}}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_summary.bootstrap_f2.R
#'
#' @export

summary.bootstrap_f2 <- function(object, ...) {
  mf <- match.call(expand.dots = TRUE)
  m <- match("digits", names(mf), 0L)

  if (m == 0) {
    digits <- getOption("digits")
  } else {
    digits <- mf[[m]]
  }

  print(object$Boot, ...)

  cat("\n\n")
  print(object$CI, ...)

  cat("\n\nShah's lower 90% BCa confidence interval:\n",
      formatC(as.numeric(object$Shah_BCa_CI[1]), digits = digits),
      "\n\n")

  invisible(object)
}

#' Print a summary of the bootstrap f2 simulation
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{bootstrap_f2}}.
#'
#' @param x An object of class \sQuote{\code{bootstrap_f2}} returned by the
#'   \code{\link{bootstrap_f2}()} function.
#' @inheritParams summary.bootstrap_f2
#'
#' @return The \sQuote{\code{bootstrap_f2}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @inherit summary.bootstrap_f2 details seealso
#'
#' @example man/examples/examples_print.bootstrap_f2.R
#'
#' @export

print.bootstrap_f2 <- function(x, ...) {

  summary(object = x, ...)

  invisible(x)
}

#' Summary of MIMCR estimation
#'
#' This is a method for the function \code{summary()} for objects of class
#' \sQuote{\code{mimcr}}.
#'
#' @param object An object of class \sQuote{\code{mimcr}} returned by the
#'   \code{\link{mimcr}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[base]{formatC}()} function.
#'
#' @details The most relevant information in an \sQuote{\code{mimcr}} object
#' is printed.
#'
#' @return The \sQuote{\code{mimcr}} object passed to the \code{object}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{mimcr}}, \code{\link[base]{formatC}},
#' \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_summary.mimcr.R
#'
#' @export

summary.mimcr <- function(object, ...) {

  cat("\nResults of Model-Independent Multivariate Confidence Region (MIMCR)",
      "\napproach to assess equivalence of highly variable in-vitro",
      "\ndissolution profiles of two drug product formulations")

  cat("\n\nDid the Newton-Raphson search converge? ")
  if (!is.na(object[["NR.CI"]][["converged"]])) {
    cat(c("No", "Yes")[object[["NR.CI"]][["converged"]]  + 1])
  } else {
    cat("NA")
  }

  cat("\nAre the points located on the confidence region boundary (CRB)? ")
  if (!is.na(object[["NR.CI"]][["points.on.crb"]])) {
    cat(c("No", "Yes")[object[["NR.CI"]][["points.on.crb"]]  + 1])
  } else {
    cat("NA")
  }

  cat("\n\nParameters (general):",
      "\nSignificance level:                  ",
      formatC(unname(object[["Parameters"]]["alpha"]), ...),
      "\nDegrees of freedom (1):              ",
      formatC(unname(object[["Parameters"]]["df1"]), ...),
      "\nDegrees of freedom (2):              ",
      formatC(unname(object[["Parameters"]]["df2"]), ...),
      "\nMahalanobis distance (MD):           ",
      formatC(unname(object[["Parameters"]]["dm"]), ...),
      "\n(F) scaling factor K:                ",
      formatC(unname(object[["Parameters"]]["K"]), ...),
      "\n(MD) scaling factor k:               ",
      formatC(unname(object[["Parameters"]]["k"]), ...),
      "\nHotelling's T2:                      ",
      formatC(unname(object[["Parameters"]]["T2"]), ...))

  cat("\n\nParameters specific for Tsong (1996) approach:",
      "\nMaximum tolerable average difference:",
      formatC(unname(object[["Parameters"]]["MTAD"]), ...),
      "\nSimilarity limit:                    ",
      formatC(unname(object[["Parameters"]]["Sim.Limit"]), ...),
      "\nObserved upper limit:                ",
      formatC(unname(object[["Parameters"]]["Obs.U"]), ...))

  cat("\n\nParameters specific for Hoffelder (2016) approach:",
      "\nNoncentrality parameter:             ",
      formatC(unname(object[["Parameters"]]["ncp.Hoffelder"]), ...),
      "\nCritial F (Hoffelder):               ",
      formatC(unname(object[["Parameters"]]["F.crit.Hoffelder"]), ...),
      "\nProbability p (Hoffelder):           ",
      formatC(unname(object[["Parameters"]]["p.F.Hoffelder"]), ...))

  cat("\n\nConclusions:",
      "\n      Tsong (1996): ", unname(object$Similarity["Tsong"]),
      "\n  Hoffelder (2016): ", unname(object$Similarity["Hoffelder"]),
      "\n\n")

  invisible(object)
}

#' Print a summary of MIMCR estimation
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{mimcr}}.
#'
#' @param x An object of class \sQuote{\code{mimcr}} returned by the
#'   \code{\link{mimcr}()} function.
#' @inheritParams summary.mimcr
#'
#' @return The \sQuote{\code{mimcr}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @inherit summary.mimcr details seealso
#'
#' @example man/examples/examples_print.mimcr.R
#'
#' @export

print.mimcr <- function(x, ...) {

  summary(object = x, ...)

  invisible(x)
}

#' Plot of the mztia simulation
#'
#' This is a method for the function \code{plot()} for objects of class
#' \sQuote{\code{plot_mztia}}.
#'
#' @param x An object of class \sQuote{\code{plot_mztia}} returned by the
#'   \code{\link{plot_mztia}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[boot]{plot.boot}()} function.
#'
#' @details The element \code{Graph} of the \sQuote{\code{plot_mztia}} object
#' that is returned by the function \code{\link{plot_mztia}()} is an object
#' of class \sQuote{\code{ggplot}}, generated by the function
#' \code{\link[ggplot2]{ggplot}()} from the \sQuote{\code{ggplot2}} package.
#' Thus, the corresponding \code{plot} method is used for plotting. Arguments
#' to the \code{\link[ggplot2]{ggplot}()} function can be passed via the
#' \code{...} parameter.
#'
#' @return The \sQuote{\code{plot_mztia}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{mztia}}, \code{\link{plot_mztia}},
#' \code{\link[ggplot2]{ggplot}()}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_plot.plot_mztia.R
#'
#' @export

plot.plot_mztia <- function(x, ...) {

  plot(x$Graph, ...)

  invisible(x)
}

#' Print a plot of MZTIA estimation
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{plot_mztia}}.
#'
#' @param x An object of class \sQuote{\code{plot_mztia}} returned by the
#'   \code{\link{plot_mztia}()} function.
#' @inheritParams plot.plot_mztia
#'
#' @inherit plot.plot_mztia details return seealso
#'
#' @example man/examples/examples_print.plot_mztia.R
#'
#' @export

print.plot_mztia <- function(x, ...) {

  plot.plot_mztia(x = x, ...)

  invisible(x)
}

#' Summary of MZTIA estimation
#'
#' This is a method for the function \code{summary()} for objects of class
#' \sQuote{\code{mztia}}.
#'
#' @param object An object of class \sQuote{\code{mztia}} returned by the
#'   \code{\link{mztia}()} function.
#' @param ...  Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[base]{print.data.frame}()}
#'   function.
#'
#' @details The \dQuote{limits} subset (see column \dQuote{frame}) of the data
#' frame that is contained in the \dQuote{Data} element of the
#' \sQuote{\code{mztia}} object is printed.
#'
#' @return The \sQuote{\code{mztia}} object passed to the \code{object}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{mztia}}, \code{\link[base]{print.data.frame}},
#' \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_summary.mztia.R
#'
#' @export

summary.mztia <- function(object, ...) {

  cat("\nResults of Martinez & Zhao Tolerance Interval (TI) Approach")
  cat("\n(TI limits calculated at each time point of the",
      "dissolution profiles of a set of reference batches)\n\n")

  print(x = object[["Limits"]], ...)

  cat("\nAbbreviations:\n")
  cat("TL: Tolerance Interval Limit (TL);",
      "LTL: lower TL;",
      "UTL: upper TL;",
      paste0("\nS1: level 1 boundary (LTL - ", object[["Variables"]]$qs[1],
             ") or (UTL + ",  object[["Variables"]]$qs[1], ");"),
      paste0("\nS2: level 2 boundary (LTL - ", object[["Variables"]]$qs[2],
             ") or (UTL + ",  object[["Variables"]]$qs[2], ").\n\n"))

  invisible(object)
}

#' Print a summary of MZTIA estimation
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{mztia}}.
#'
#' @param x An object of class \sQuote{\code{mztia}} returned by the
#'   \code{\link{mztia}()} function.
#' @inheritParams summary.mztia
#'
#' @return The \sQuote{\code{mztia}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @inherit summary.mztia details seealso
#'
#' @example man/examples/examples_print.mztia.R
#'
#' @export

print.mztia <- function(x, ...) {

  summary(object = x, ...)

  invisible(x)
}
