% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comprehension.R
\name{comprehension}
\alias{comprehension}
\alias{Comp}
\alias{List}
\alias{Env}
\alias{Vec}
\alias{Num}
\alias{Chr}
\alias{Logical}
\alias{Mat}
\alias{DF}
\title{Vectorized Comprehension in R}
\usage{
Comp(map = lapply, fun = NULL)

List(loop, clust = NULL, fun = NULL)

Env(loop, clust = NULL)

Vec(loop, clust = NULL, drop.names = FALSE)

Num(loop, clust = NULL, drop.names = FALSE)

Chr(loop, clust = NULL, drop.names = FALSE)

Logical(loop, clust = NULL, drop.names = FALSE)

Mat(loop, clust = NULL, by.col = TRUE)

DF(loop, clust = NULL)
}
\arguments{
\item{map}{function, such as \code{lapply}, that is used for the comprehension}

\item{fun}{function to be called on result after comprehension}

\item{loop}{a \code{for} loop with format: \code{for (var in seq) <name => <if (cond)> expr}. See "details" below.}

\item{clust}{cluster to use for \code{\link[parallel:clusterApply]{parallel}} computations}

\item{drop.names}{logical; should names be dropped after conversion? Defaults to \code{FALSE}.}

\item{by.col}{should comprehension on matrix group by columns or rows? Defaults to \code{TRUE}.}
}
\value{
Determined by the function. \code{List} returns an object of class 'list',
\code{Num} returns a numeric vector, etc. See the descriptions of each function for
their return type.
}
\description{
Functions that provide Python-style list (and related) comprehension.
Comprehensions convert \code{\link[base:Control]{for}} loops into \code{\link[base:lapply]{lapply}} functions
before evaluation. Support for multiple variables, name assignment, nested loops,
custom iterators, if-else statements, and variety of return types included.
}
\details{
The comprehension functions parse an R loop expression into \code{\link[base:lapply]{lapply}}
functions to allow for more readable code and easy creation and conversion
of vectors. The general syntax for a loop expression is as follows:

\code{for (var in seq) <name=> <if (cond)> expr}

where \code{<...>} denotes optional statements. The \code{seq} can be any R object:
a list, matrix, data.frame, environment, function, etc. The function \code{\link{iter}}
is called on the \code{seq}. So the behavior can be easily described for custom
classes or objects. See \code{\link{helpers}} for functions like \code{\link{zip}}
that can be used with \code{seq}.

Multiple variables can be used in \code{var} by separating the names with a period ".".
For example, \code{i.j} is equivalent looping with variables \code{i} and \code{j}.
The downside is that periods cannot be used in the \code{var} name. When multiple variables
are used, the object received from the sequence at each iteration is split and its
elements assigned in order to each of the variables. If the \code{var} is \code{i.j} and
the object received in the iteration was \code{c(2,4,6)}, then \code{i=2}, \code{j=4},
and 6 would not be assigned. Since variables are split on periods, \code{i..j} could
be used to assign the first and third elements, or \code{.i.j} the second and third.
Any number of variables can be used. Note that the entire object is returned if
there are no periods in the name, so use \code{i..} if only the first object is needed.

To provide names within a loop, preface the expression with the desired \code{name} for
that particular object followed by \code{=}. \code{name} can be any expression, just
make sure to surround any \code{if} chain for the name with parentheses, or the R
parser will not detect that the assignment operator is associated with the \code{expr}.
Behind the scenes, the expression on the left-hand side of \code{"="} is wrapped in
an \code{sapply} function and the results are assigned to the \code{\link[base:names]{names}}
of the right-hand side result.

The \code{if} statement can contain any number of \code{if-else} statements and can
be nested. Similarly, \code{for} statements can be nested any number of times and
converted to \code{lapply} as long as the expression is a self-contained \code{for} loop.

Though comprehensions are functions, both \code{List(for ...)} and \code{List[for ...]}
syntax are supported. See \code{\link{..}} for a convenience wrapper around \code{Vec}.

The different comprehensions primarily describe the return value, with \code{List}
return a "list" and \code{Num} returning a numeric vector. If the object cannot be
converted, then an error will be produced. For \code{Env}, the objects must be
named. This means that either the name must be assigned within the loop or the
loop is performed across a named object and the name is preserved. Another
difference is that is some comprehensions - though related to atomic vectors -
convert \code{for} to \code{sapply} while others convert to \code{lapply}.

The \code{Comp} function is used to create custom comprehensions. It should be
supplied with a \code{map} function such as \code{\link[base:lapply]{lapply}} that
accepts arguments: \code{X} for the argument over which the comprehension
iterates, \code{FUN} a function applied to each element, and \code{...} for
additional arguments passed to the \code{FUN}. \code{Comp} also accepts a
post-evaluation function, \code{fun}, that is applied to the result. This
could be used to ensure that the result complies to some class or other
restriction.

Users can also specify a cluster to use. If specified, then a parallel version
of \code{lapply} or \code{sapply} is used based on \code{parLapply} and \code{parSapply}
from the \code{\link[parallel:clusterApply]{parallel}} package. This can greatly
reduce the calculation time for different operations, but has additional overhead
that makes the cost greater than the benefit for relatively small vectors. See
\code{\link{auto_cluster}} for auto-creation.
}
\section{Functions}{
\itemize{
\item \code{Comp}: Create generalized comprehension function

\item \code{List}: Generate a 'list' from a \code{for} loop

\item \code{Env}: Generate an 'environment' from a \code{for} loop

\item \code{Vec}: Generate a flat, atomic 'vector' from a \code{for} loop

\item \code{Num}: Generate a 'numeric' vector from a \code{for} loop

\item \code{Chr}: Generate a 'character' vector from a \code{for} loop

\item \code{Logical}: Generate a 'logical' vector from a \code{for} loop

\item \code{Mat}: Generate a 'matrix' from a \code{for} loop

\item \code{DF}: Generate a 'data.frame' from a \code{for} loop
}}

\examples{
people <- list(
  John = list(age = 30, weight = 180, mood = "happy", gender = "male"),
  April = list(age = 26, weight = 110, mood = "sad", gender = "female"),
  Jill = list(age = 42, weight = 125, mood = "ok", gender = "female")
)

weight_kg <- Num(for (i in people) i$weight/2.2)
gender <- Chr(for (i in people) i$gender)
gender_tab <- List(for (i in c("male", "female")) i = length(which(gender == i)))

Chr(for (..i.j in people) paste0(i, " & ", j))

Chr(for (i.j in items(people)) paste0(i, " is ", j$age, " years old."))

e <- Env(for (i.j in items(people)) i = j$age)
e$John

Num(for (i in 1:10) for (j in 2:6) if (i == j) i^2)

}
