\name{weights.enetLTS}
\alias{weights.enetLTS}
%-------------------------------------------------
\title{
 binary weights from the \code{"enetLTS"} object
}
%-------------------------------------------------
\description{
Extract binary weights that indicate outliers from the current model.
}
%-------------------------------------------------
\usage{
\method{weights}{enetLTS}(object,vers=c("reweighted","raw","both"),index=FALSE,...)
}
%-------------------------------------------------
\arguments{
 \item{object}{the model fit from which to extract outlier weights.}

\item{vers}{a character string specifying for which estimator to extract
outlier weights. Possible values are \code{"reweighted"} (the default) for
weights indicating outliers from the reweighted fit, \code{"raw"} for
weights indicating outliers from the raw fit, or \code{"both"} for the
outlier weights from both estimators.}

\item{index}{a logical indicating whether the indices of the weight vector should
  be included or not (the default is \code{FALSE}).}

\item{\dots}{additional arguments from the \code{enetLTS} object if needed.}
}
%-------------------------------------------------
\value{
A numeric vector containing the requested outlier weights.
}
%-------------------------------------------------
\note{
The weights are \eqn{1} for observations with reasonably small
residuals and \eqn{0} for observations with large residuals.
Here, residuals represent standardized residuals
for \code{family="gaussian"}, Pearson residuals for
\code{family="binomial"} and group-wise scaled robust distances
\code{family="multinomial"}.

Use weights with or without index is available.
}
%-------------------------------------------------
\seealso{
\code{\link{enetLTS}}
}
%-------------------------------------------------
\examples{
## for gaussian

set.seed(86)
n <- 100; p <- 25                             # number of observations and variables
beta <- rep(0,p); beta[1:6] <- 1              # 10\% nonzero coefficients
sigma <- 0.5                                  # controls signal-to-noise ratio
x <- matrix(rnorm(n*p, sigma),nrow=n)
e <- rnorm(n,0,1)                             # error terms
eps <- 0.1                                    # contamination level
m <- ceiling(eps*n)                           # observations to be contaminated
eout <- e; eout[1:m] <- eout[1:m] + 10        # vertical outliers
yout <- c(x \%*\% beta + sigma * eout)        # response
xout <- x; xout[1:m,] <- xout[1:m,] + 10      # bad leverage points

\donttest{
fit1 <- enetLTS(xout,yout)
weights(fit1)
weights(fit1,vers="raw",index=TRUE)
weights(fit1,vers="both",index=TRUE)}


## for binomial

eps <-0.05                                     # \%10 contamination to only class 0
m <- ceiling(eps*n)
y <- sample(0:1,n,replace=TRUE)
xout <- x
xout[y==0,][1:m,] <- xout[1:m,] + 10;          # class 0
yout <- y                                      # wrong classification for vertical outliers

\donttest{
fit2 <- enetLTS(xout,yout,family="binomial")
weights(fit2)
weights(fit2,vers="raw",index=TRUE)
weights(fit2,vers="both",index=TRUE)
}

## for multinomial

n <- 120; p <- 15
NC <- 3
X <- matrix(rnorm(n * p), n, p)
betas <- matrix(1:NC, ncol=NC, nrow=p, byrow=TRUE)
betas[(p-5):p,]=0; betas <- rbind(rep(0,NC),betas)
lv <- cbind(1,X) \%*\% betas
probs <- exp(lv)/apply(exp(lv),1,sum)
y <- apply(probs,1,function(prob){sample(1:NC, 1, TRUE, prob)})
xout <- X
eps <-0.05                          # \%10 contamination to only class 0
m <- ceiling(eps*n)
xout[1:m,] <- xout[1:m,] + 10       # bad leverage points
yout <- y

\donttest{
fit3    <- enetLTS(xout,yout,family="multinomial")
weights(fit3)
weights(fit3,vers="raw",index=TRUE)
weights(fit3,vers="both",index=TRUE)
}
}
%-------------------------------------------------
\author{
Fatma Sevinc KURNAZ, Irene HOFFMANN, Peter FILZMOSER
\cr Maintainer: Fatma Sevinc KURNAZ <fatmasevinckurnaz@gmail.com>;<fskurnaz@yildiz.edu.tr>}
%-------------------------------------------------
\keyword{regression}
\keyword{classification}


