% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ergmito-methods.R, R/ergmito.R
\name{vcov.ergmito}
\alias{vcov.ergmito}
\alias{ergmito}
\title{Estimation of ERGMs using Maximum Likelihood Estimation (MLE)}
\usage{
\method{vcov}{ergmito}(object, solver = NULL, ...)

ergmito(
  model,
  model_update = NULL,
  stats_weights = NULL,
  stats_statmat = NULL,
  optim.args = list(),
  init = NULL,
  use.grad = TRUE,
  target_stats = NULL,
  ntries = 1L,
  keep.stats = TRUE,
  target_offset = NULL,
  stats_offset = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{ergmito}}

\item{solver}{Function. Used to compute the inverse of the hessian matrix. When
not null, the variance-covariance matrix is recomputed using that function.
By default, \code{ergmito} uses \link[MASS:ginv]{MASS::ginv}.}

\item{...}{Further arguments passed to the method. In the case of \code{ergmito},
\code{...} are passed to \link{ergmito_formulae}.}

\item{model}{Model to estimate. See \link[ergm:ergm]{ergm::ergm}. The only difference with
\code{ergm} is that the LHS can be a list of networks.}

\item{model_update}{A \code{\link[stats:formula]{formula}}. this can be used to
apply transformations, create interaction effects, add offset terms, etc.
(see examples below and more details in \link{ergmito_formulae}).}

\item{stats_weights}{Either an integer vector or a list of integer vectors (see \link{exact_loglik}).}

\item{stats_statmat}{Either a matrix or a list of matrices (see \link{exact_loglik}).}

\item{optim.args}{List. Passed to \link[stats:optim]{stats::optim}.}

\item{init}{A numeric vector. Sets the starting parameters for the
optimization routine. Default is a vector of zeros.}

\item{use.grad}{Logical. When \code{TRUE} passes the gradient function to \code{optim}.
This is intended for testing only (internal use).}

\item{target_stats}{A matrix of target statistics (see \link[ergm:ergm]{ergm::ergm}).}

\item{ntries}{Integer scalar. Number of tries to estimate the MLE (see details).}

\item{keep.stats}{Logical scalar. When \code{TRUE} (the default), the matrices
and vectors associated with the sufficient statistics will be returned.
Otherwise the function discards them. This may be useful for saving memory
space when estimating multiple models.}

\item{target_offset, stats_offset}{See \code{\link[=exact_loglik]{exact_loglik()}}.}
}
\value{
An list of class \code{ergmito}:
\itemize{
\item \code{call}          The program call.
\item \code{coef}          Named vector. Parameter estimates.
\item \code{iterations}    Integer. Number of times the log-likelihood was evaluated
(see \link[stats:optim]{stats::optim}).
\item \code{mle.lik}       Numeric. Final value of the objective function.
\item \code{null.lik}      Numeric. Final value of the objective function for the null model.
\item \code{covar}         Square matrix of size \code{length(coef)}. Variance-covariance matrix
computed using the exact hessian as implemented in \link{exact_hessian}.
\item \code{coef.init}     Named vector of length \code{length(coef)}. Initial set of parameters
used in the optimization.
\item \code{formulae}      An object of class \link[=ergmito_formulae]{ergmito_loglik}.
\item \code{nobs}          Integer scalar. Number of networks in the model.
\item \code{network}       Networks passed via \code{model}.
\item \code{optim.out},\code{optim.args} Results from the optim call and arguments passed to it.
\item \code{status},\code{note} Convergence code. See \link{check_convergence}
\item \code{best_try}      Integer scalar. Index of the run with the highest log-likelihood value.
\item \code{history}       Matrix of size \code{ntries * (k + 1)}. History of the parameter
estimates and the reached log-likelihood values.
\item \code{timer}         Vector of times (for benchmarking). Each unit marks the starting
point of the step.
}

Methods \code{\link[base:print]{base::print()}}, \code{\link[base:summary]{base::summary()}}, \code{\link[stats:coef]{stats::coef()}}, \code{\link[stats:logLik]{stats::logLik()}},
\code{\link[stats:nobs]{stats::nobs()}}, \code{\link[stats:vcov]{stats::vcov()}}, \code{\link[stats:AIC]{stats::AIC()}}, \code{\link[stats:AIC]{stats::BIC()}},
\code{\link[stats:confint]{stats::confint()}}, and  \code{\link[stats:formula]{stats::formula()}} are available.
}
\description{
\code{ergmito} uses Maximum Likelihood Estimation (MLE) to fit Exponential Random
Graph Models for single or multiple small networks, the later using
pooled-data MLE. To do so we use exact likelihoods, which implies fully
enumerating the support of the model. Overall, the exact likelihood
calculation is only possible when dealing with directed (undirected) networks
size 5 (7). In general, directed (undirected) graphs with more than 5 (7)
vertices should not be fitted using MLE, but instead other methods such as
the MC-MLE algorithm or the Robbins-Monro Stochastic Approximation algorithm,
both of which are available in the ergm R package.The workhorse function of
\code{ergmito} is the \code{ergm} package function \code{\link[ergm:ergm.allstats]{ergm::ergm.allstats()}}.
}
\details{
The support of the sufficient statistics is calculated using ERGM's
\code{\link[ergm:ergm.allstats]{ergm::ergm.allstats()}} function.
}
\section{MLE}{


Maximum Likelihood Estimates are obtained using the \link[stats:optim]{stats::optim} function.
The default method for maximization is \code{BFGS} using both the log-likelihood
function and its corresponding gradient.

Another important factor to consider is the existence of the MLE estimates
As shown in Handcock (2003), if the observed statistics are near the border
if the support function (e.g. too many edges or almost none), then, even if
the MLE estimates exists, the optimization function may not be able to reach
the optima. Moreover, if the target (observed) statistics live in the boundary,
then the MLE estimates do not exists. In general, this should not be an issue
in the context of the pooled model, as the variability of observed statistics
should be enough to avoid those situations.

The function \code{ergmito} will try to identify possible cases of non-existence,
of the MLE, and if identified then try to re estimate the model parameters using
larger values than the ones obtained, if the log-likelihood is greater, then it is
assumed that the model is degenerate and the corresponding values will be
replaced with either \code{+Inf} or  \code{-Inf}. By default, this behavior is checked
anytime that the absolute value of the estimates is greater than 5, or the
sufficient statistics were flagged as potentially outside of the interior of
the support (close to zero or to its max).

In the case of \code{ntries}, the optimization is repeated that number of times,
each time perturbing the \code{init} parameter by adding a Normally distributed
vector. The result which reaches the highest log-likelihood will be the one
reported as parameter estimates. This feature is intended for testing only.
Anecdotally, \code{optim} reaches the max in the first try.
}

\examples{

# Generating a small graph
set.seed(12)
n <- 4
net <- rbernoulli(n, p = .3)

model <- net ~ edges + mutual

library(ergm)
ans_ergmito <- ergmito(model)
ans_ergm  <- ergm(model)

# The ergmito should have a larger value
ergm.exact(ans_ergmito$coef, model)
ergm.exact(ans_ergm$coef, model)

summary(ans_ergmito)
summary(ans_ergm)

# Example 2: Estimating an ERGMito using data with know DGP parameters -----
data(fivenets) 

model1 <- ergmito(fivenets ~ edges + nodematch("female"))
summary(model1) # This data has know parameters equal to -2.0 and 2.0

# Example 3: Likelihood ratio test using the lmtest R package

if (require(lmtest)) {
  data(fivenets)
  model1 <- ergmito(fivenets ~ edges + nodematch("female"))
  model2 <- ergmito(fivenets ~ edges + nodematch("female") + mutual)
  
  lrtest(model1, model2)
  # Likelihood ratio test
  # 
  # Model 1: fivenets ~ edges + nodematch("female") 
  # Model 2: fivenets ~ edges + nodematch("female") + mutual
  #   #Df  LogLik Df  Chisq Pr(>Chisq)
  # 1   2 -34.671                     
  # 2   3 -34.205 1 0.9312     0.3346
}

# Example 4: Adding an reference term for edge-count ----------------------

# Simulating networks of different sizes
set.seed(12344)
nets <- rbernoulli(c(rep(4, 10), rep(5, 10)), c(rep(.2, 10), rep(.1, 10)))

# Fitting an ergmito under the Bernoulli model
ans0 <- ergmito(nets ~ edges)
summary(ans0)
# 
# ERGMito estimates
# 
# formula:
#   nets ~ edges
# 
#       Estimate Std. Error z value  Pr(>|z|)    
# edges -1.68640    0.15396 -10.954 < 2.2e-16 ***
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# AIC: 279.3753    BIC: 283.1436    (Smaller is better.) 


# Fitting the model including a reference term for networks of size 5.
# Notice that the variable -n- and other graph attributes can be used
# with -model_update-.
ans1 <- ergmito(nets ~ edges, model_update = ~ I(edges * (n == 5)))
summary(ans1)
# 
# ERGMito estimates
# 
# formula:
#   nets ~ edges + I(edges * (n == 5))
# 
#                     Estimate Std. Error z value  Pr(>|z|)    
# edges               -1.18958    0.21583 -5.5116 3.556e-08 ***
# I(edges * (n == 5)) -0.90116    0.31250 -2.8837   0.00393 ** 
# ---
# Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
# AIC: 272.9916    BIC: 280.5282    (Smaller is better.) 

# The resulting parameter for the edge-count is smaller for networks
# of size five
plogis(coef(ans1)[1])   # 0.23
plogis(sum(coef(ans1))) # 0.11

# We can see that in this case the difference in edge-count matters.
if (require(lmtest)) {

  lrtest(ans0, ans1)
  # Likelihood ratio test
  # 
  # Model 1: nets ~ edges
  # Model 2: nets ~ edges + I(edges * (n == 5))
  # #Df  LogLik Df  Chisq Pr(>Chisq)   
  # 1   1 -138.69                        
  # 2   2 -134.50  1 8.3837   0.003786 **
  #   ---
  #   Signif. codes:  0 ‘***’ 0.001 ‘**’ 0.01 ‘*’ 0.05 ‘.’ 0.1 ‘ ’ 1
}

}
\seealso{
The function \code{\link[=plot.ergmito]{plot.ergmito()}} and \code{\link[=gof_ergmito]{gof_ergmito()}} for post-estimation
diagnostics.
}
