#' esList
#
#' @description esList holds all separate raw ESM datasets in one list, which is an R-built-in data structure.
#
#' @param dfList a list. Each element of the list must be a data.frame. Each data.frame is a separate raw ESM dataset/an ESM questionnaire version. If there is just one ESM version the list therefore contains one data.frame.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @details The separate raw ESM datasets are still separated in the list. Each list element is named according to the survey version as specified by the content of the respective column name in each ESM dataset.
#
#' @return A named list of the ESM datasets, where each list name is equal to the name of the respective ESM questionnaire version.
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute esList. Start --------------------
#' # Generate argument RELEVANTVN_ES
#' relEs <- relevantESVN(svyName="survey_name", IMEI="IMEI",
#' STARTDATE="start_date", STARTTIME="start_time",
#' ENDDATE="end_date", ENDTIME="end_time")
#' imeiNumbers <- as.character(referenceDf$imei)
#' surveyNames <- c("morningTestGroup", "dayTestGroup", "eveningTestGroup",
#' "morningControlGroup", "dayControlGroup", "eveningControlGroup")
#' RELEVANT_ES <- setES(4, imeiNumbers, surveyNames, relEs)
#' RELEVANTVN_ES <- RELEVANT_ES[["RELEVANTVN_ES"]]
#' # Prerequisites in order to execute esList. End ----------------------
#' # ------------------------------------------------------
#' # Run function 6 of 29; see esmprep functions' hierarchy.
#' # ------------------------------------------------------
#' # 6 exemplary raw ESM (sub-)datasets.
#' esLs <- esList(list(morningControl, dayControl, eveningControl,
#' morningTest, dayTest, eveningTest), RELEVANTVN_ES)
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 5 of 29).
#
#' @export
#
esList <- function(dfList, RELEVANTVN_ES=NULL) {
    
    # Check whether all elements of dfList are of class data.frame.
    dfCheck <- sapply(dfList, FUN = is.data.frame)
    
    # If at least one of the elements in dfList is not of type data.frame
    # stop.
    if(any(dfCheck == FALSE)) {
        stop("Error in argument 'dfList'. It must be of type 'list'. All elements in the list must be of type 'data.frame'.")
    }
    
    # Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=NULL,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=NULL)
    
    # 1. If [there is exactly one survey version and] there is no column holding the
    # survey name (meaning that the list element ES_SVY_NAME of the list RELEVANTVN_ES
    # contains "NO_ESM_COLUMN_PRESENT_SPECIFYING_SURVEY_NAME"), change that long string
    # to a much shorter string and use as column name.
    if(RELEVANTVN_ES[["ES_SVY_NAME"]] == "ES_SVY_NAME") {
    		# 2. Generate new column with a survey name.
    		for(svyName_i in 1:length(dfList)) {
    			if(length(dfList) == 1) {
    				dfList[[svyName_i]][,RELEVANTVN_ES[["ES_SVY_NAME"]]] <- "ESMVERSION_ESMPREP"
    			} else {
    				dfList[[svyName_i]][,RELEVANTVN_ES[["ES_SVY_NAME"]]] <- paste0("ESMVERSION", svyName_i, "_ESMPREP")
    			}
    		}
    }
    
    # Empty variable that collects the names of the list elements of
    # esList's return value.
    seqVersionNames <- c()
    for(i in 1:length(dfList)) {
        
        # surveyNameDontExist contains all unique character strings that
        # are contained in the column, which has been passed by the user
        # as the column that contains the name of the ESM version.
        surveyNameDontExist <- tryCatch(
            {unique(dfList[[i]][,RELEVANTVN_ES[["ES_SVY_NAME"]]])},
            # If there is no character string in the variable then
            # "surveyNameNotFound" is returned.
            error = function(e) {"surveyNameNotFound"})
        
        # If the tryCatch function returned "surveyNameNotFound", stop.
        if(any(as.character(surveyNameDontExist[!is.na(surveyNameDontExist)]) == "surveyNameNotFound")) {
            stop("Column name for survey version not found in data.frame.")
        } else {
            # Logical vector if any NAs should be among the survey version names
            isNA_svyVersion <- is.na(unique(as.character(dfList[[i]] [,RELEVANTVN_ES[["ES_SVY_NAME"]]])))
            # Extract all unique characters with more than one letter.
            versionName <- unique(as.character(dfList[[i]] [,RELEVANTVN_ES[["ES_SVY_NAME"]]]))[!isNA_svyVersion]
        }
        
        # Since the survey version name will be the list name of the
        # esList's output, there must be no white spaces or anything
        # unusual in that name.
        
        # Remove any white spaces
        versionName <- gsub(" ", "", versionName)
        # emptyString = string consisting only of white spaces
        emptyString <- versionName == ""
        
        # Select the name(s) that consist of at least one letter.
        versionName <- versionName[!is.na(versionName) & !emptyString]
        
        # There must not be more than one character string which
        # must have at least have one letter.
        if(length(versionName) != 1 & nchar(versionName) > 1) {
            stop(paste0("Each data.frame must represent exactly one event sampling questionnaire version, i.e. the column ", RELEVANTVN_ES[["ES_SVY_NAME"]], " must contain the respective version name."))
        }
        # Append to the collecting variable.
        seqVersionNames <- c(seqVersionNames, versionName)
    }
    
    # Append the names of the respective event sampling
    # questionnaire version to the list.
    names(dfList) <- seqVersionNames
    dfList
}
