% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survHE_funcs.R
\name{fit.models.expert}
\alias{fit.models.expert}
\title{Fit Parametric Survival Models Incorporating Expert Opinion}
\usage{
fit.models.expert(
  formula = NULL,
  data,
  distr = NULL,
  method = "bayes",
  opinion_type = "survival",
  param_expert = NULL,
  ...
)
}
\arguments{
\item{formula}{An object of class \code{\link{formula}} specifying the survival model to be fitted, as per \code{fit.models} in the \pkg{survHE} package. The left-hand side must be a \code{\link[survival]{Surv}} object, and the right-hand side specifies the covariates.}

\item{data}{A data frame containing the variables specified in the \code{formula}, as per \code{fit.models}.}

\item{distr}{A character vector specifying the distribution(s) to be used for the survival model(s), as per \code{fit.models}. Options include, but are not limited to:
\itemize{
\item \code{"exp"}: Exponential distribution
\item \code{"wei"}: Weibull distribution
\item \code{"gom"}: Gompertz distribution
\item \code{"gengamma"}: Generalized Gamma distribution
\item \code{"genf"}: Generalized F distribution (not available for \code{method = "bayes"})
\item \code{"rps"}: Royston-Parmar spline model (not available with \code{opinion_type = "mean"})
}
Note: The Generalized F model is not available when \code{method = "bayes"}, and the Royston-Parmar model is not available with expert opinion on the mean survival.}

\item{method}{The estimation method to be used. Options are:
\itemize{
\item \code{"mle"}: Maximum Likelihood Estimation
\item \code{"bayes"}: Bayesian estimation using either Stan or JAGS
}
Note: The \code{"inla"} method is not included. For Bayesian analysis, specify \code{method = "bayes"} (do not use \code{"hmc"} as in \pkg{survHE}).}

\item{opinion_type}{A character string specifying the type of expert opinion provided:
\itemize{
\item \code{"survival"}: Expert opinion on the survival function at specific time points
\item Other values (e.g., \code{"mean"}): Expert opinion on differences in expected survival (area under the survival curve)
}}

\item{param_expert}{A list where each element corresponds to a time point (if applicable) and contains a data frame of expert opinions. Each data frame should have the following columns, with each row representing an expert:
\describe{
\item{\code{dist}}{Name of the distribution assigned to each expert's opinion. Options include \code{"norm"}, \code{"t"}, \code{"lnorm"}, \code{"gamma"}, \code{"beta"}.}
\item{\code{wi}}{Weight of the expert's opinion. Weights must sum to 1 across all experts for each time point.}
\item{\code{param1}}{First parameter of the specified distribution (e.g., mean for normal distribution). Parameters follow the conventions of the \pkg{SHELF} package.}
\item{\code{param2}}{Second parameter of the specified distribution (e.g., standard deviation for normal distribution).}
\item{\code{param3}}{Third parameter of the distribution, if applicable (e.g., degrees of freedom for the t-distribution); otherwise, set to \code{NA}.}
}}

\item{...}{Other arguments required depending on the analysis. Important ones include:
\describe{
\item{\code{id_St}}{Required if the model includes covariates (e.g., treatments) and expert opinion on survival probabilities. Specifies the row number in the data frame representing the covariate pattern for which the expert opinion is provided.}
\item{\code{id_trt}}{Required if including expert opinion about differences in expected survival. Specifies the row number representing the treatment group.}
\item{\code{id_comp}}{Required if including expert opinion about differences in expected survival. Specifies the row number representing the comparator group.}
\item{\code{times_expert}}{A numeric vector of time points at which expert opinion on survival probabilities is provided.}
\item{\code{compile_mods}}{For Bayesian analysis, a list of pre-compiled Stan models can be supplied to speed up computation. Pre-compiling models is recommended and can be done using \code{\link{compile_stan}}.}
}}
}
\value{
An object of class \code{expertsurv} containing the fitted models, parameter estimates, and other relevant information. This object can be used with plotting and summary functions for further analysis.
}
\description{
The \code{fit.models.expert} function extends the capabilities of the \code{survHE} package by allowing users to fit parametric survival models that incorporate expert opinion. Expert opinions can be on survival probabilities at specific time points or on expected differences in survival between groups. This function is particularly useful when empirical data is scarce or incomplete, and expert knowledge can help inform the analysis.
}
\details{
This function enables the integration of expert opinion into parametric survival models. Expert opinion can be particularly valuable when data is limited or censored, as it allows for informed estimates of survival functions or differences between treatment groups.

The function supports both Maximum Likelihood Estimation (MLE) and Bayesian methods. For Bayesian estimation, models are fitted using Stan or JAGS, depending on the distribution. Pre-compiling Stan models using \code{\link{compile_stan}} is highly recommended to reduce computation time.
}
\examples{
\dontrun{
library(dplyr)

# Example 1: Incorporating Expert Opinion on Survival Probabilities Using MLE

# Define expert opinion as a normal distribution centered at 0.1 with sd 0.05
param_expert_example1 <- list()
param_expert_example1[[1]] <- data.frame(
  dist = "norm",
  wi = 1,  # Ensure weights sum to 1 across experts
  param1 = 0.1,
  param2 = 0.05,
  param3 = NA
)

# Time point at which expert opinion is provided
timepoint_expert <- 14  # For example, 14 months

# Prepare the data
# Assume 'data' is your dataset containing 'time' and 'censored' variables
data2 <- data \%>\%
  rename(status = censored) \%>\%
  mutate(
    time2 = ifelse(time > 10, 10, time),
    status2 = ifelse(time > 10, 0, status)
  )

# Fit the survival models using MLE, incorporating expert opinion
example1 <- fit.models.expert(
  formula = Surv(time2, status2) ~ 1,
  data = data2,
  distr = c("wei", "gom"),  # Weibull and Gompertz distributions
  method = "mle",
  opinion_type = "survival",
  times_expert = timepoint_expert,
  param_expert = param_expert_example1
)

# Plot the fitted survival curves along with the Kaplan-Meier estimate
plot(example1, add.km = TRUE, t = 0:20)

# Compare models using Akaike Information Criterion (AIC)
model.fit.plot(example1, type = "aic")

# Example 2: Incorporating Expert Opinion Using Bayesian Estimation

# Pre-compile Stan models (ideally after installing the package)
# This step can be time-consuming but only needs to be done once per session
compiled_models_saved <- compile_stan()

# Fit the survival models using Bayesian estimation with expert opinion
example1_bayes <- fit.models.expert(
  formula = Surv(time2, status2) ~ 1,
  data = data2,
  distr = c("wei", "gom"),
  method = "bayes",
  opinion_type = "survival",
  times_expert = timepoint_expert,
  param_expert = param_expert_example1,
  iter = 2000,  # Set to a high number for convergence (e.g., 2000 or more)
  compile_mods = compiled_models_saved
)

# Summarize the Bayesian model results
summary(example1_bayes)

# Plot the Bayesian fitted survival curves
plot(example1_bayes, add.km = TRUE, t = 0:20)
}
}
\keyword{Bayesian}
\keyword{expert-opinion}
\keyword{models}
\keyword{survival}
