% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulations.R
\name{SimulateCorrelation}
\alias{SimulateCorrelation}
\title{Simulation of a correlation matrix}
\usage{
SimulateCorrelation(
  pk = 10,
  theta = NULL,
  implementation = HugeAdjacency,
  topology = "random",
  nu_within = 0.1,
  nu_between = NULL,
  nu_mat = NULL,
  v_within = c(0.5, 1),
  v_between = c(0.1, 0.2),
  v_sign = c(-1, 1),
  continuous = TRUE,
  pd_strategy = "diagonally_dominant",
  ev_xx = NULL,
  scale_ev = TRUE,
  u_list = c(1e-10, 1),
  tol = .Machine$double.eps^0.25,
  output_matrices = FALSE,
  ...
)
}
\arguments{
\item{pk}{vector of the number of variables per group in the simulated
dataset. The number of nodes in the simulated graph is \code{sum(pk)}. With
multiple groups, the simulated (partial) correlation matrix has a block
structure, where blocks arise from the integration of the \code{length(pk)}
groups. This argument is only used if \code{theta} is not provided.}

\item{theta}{optional binary and symmetric adjacency matrix encoding the
conditional independence structure.}

\item{implementation}{function for simulation of the graph. By default,
algorithms implemented in \code{\link[huge]{huge.generator}} are used.
Alternatively, a user-defined function can be used. It must take \code{pk},
\code{topology} and \code{nu} as arguments and return a
\code{(sum(pk)*(sum(pk)))} binary and symmetric matrix for which diagonal
entries are all equal to zero. This function is only applied if
\code{theta} is not provided.}

\item{topology}{topology of the simulated graph. If using
\code{implementation=HugeAdjacency}, possible values are listed for the
argument \code{graph} of \code{\link[huge]{huge.generator}}. These are:
"random", "hub", "cluster", "band" and "scale-free".}

\item{nu_within}{probability of having an edge between two nodes belonging to
the same group, as defined in \code{pk}. If \code{length(pk)=1}, this is
the expected density of the graph. If \code{implementation=HugeAdjacency},
this argument is only used for \code{topology="random"} or
\code{topology="cluster"} (see argument \code{prob} in
\code{\link[huge]{huge.generator}}). Only used if \code{nu_mat} is not
provided.}

\item{nu_between}{probability of having an edge between two nodes belonging
to different groups, as defined in \code{pk}. By default, the same density
is used for within and between blocks (\code{nu_within}=\code{nu_between}).
Only used if \code{length(pk)>1}. Only used if \code{nu_mat} is not
provided.}

\item{nu_mat}{matrix of probabilities of having an edge between nodes
belonging to a given pair of node groups defined in \code{pk}.}

\item{v_within}{vector defining the (range of) nonzero entries in the
diagonal blocks of the precision matrix. These values must be between -1
and 1 if \code{pd_strategy="min_eigenvalue"}. If \code{continuous=FALSE},
\code{v_within} is the set of possible precision values. If
\code{continuous=TRUE}, \code{v_within} is the range of possible precision
values.}

\item{v_between}{vector defining the (range of) nonzero entries in the
off-diagonal blocks of the precision matrix. This argument is the same as
\code{v_within} but for off-diagonal blocks. It is only used if
\code{length(pk)>1}.}

\item{v_sign}{vector of possible signs for precision matrix entries. Possible
inputs are: \code{-1} for positive partial correlations, \code{1} for
negative partial correlations, or \code{c(-1, 1)} for both positive and
negative partial correlations.}

\item{continuous}{logical indicating whether to sample precision values from
a uniform distribution between the minimum and maximum values in
\code{v_within} (diagonal blocks) or \code{v_between} (off-diagonal blocks)
(if \code{continuous=TRUE}) or from proposed values in \code{v_within}
(diagonal blocks) or \code{v_between} (off-diagonal blocks) (if
\code{continuous=FALSE}).}

\item{pd_strategy}{method to ensure that the generated precision matrix is
positive definite (and hence can be a covariance matrix). If
\code{pd_strategy="diagonally_dominant"}, the precision matrix is made
diagonally dominant by setting the diagonal entries to the sum of absolute
values on the corresponding row and a constant u. If
\code{pd_strategy="min_eigenvalue"}, diagonal entries are set to the sum of
the absolute value of the smallest eigenvalue of the precision matrix with
zeros on the diagonal and a constant u.}

\item{ev_xx}{expected proportion of explained variance by the first Principal
Component (PC1) of a Principal Component Analysis. This is the largest
eigenvalue of the correlation (if \code{scale_ev=TRUE}) or covariance (if
\code{scale_ev=FALSE}) matrix divided by the sum of eigenvalues. If
\code{ev_xx=NULL} (the default), the constant u is chosen by maximising the
contrast of the correlation matrix.}

\item{scale_ev}{logical indicating if the proportion of explained variance by
PC1 should be computed from the correlation (\code{scale_ev=TRUE}) or
covariance (\code{scale_ev=FALSE}) matrix. If \code{scale_ev=TRUE}, the
correlation matrix is used as parameter of the multivariate normal
distribution.}

\item{u_list}{vector with two numeric values defining the range of values to
explore for constant u.}

\item{tol}{accuracy for the search of parameter u as defined in
\code{\link[stats]{optimise}}.}

\item{output_matrices}{logical indicating if the true precision and (partial)
correlation matrices should be included in the output.}

\item{...}{additional arguments passed to the graph simulation function
provided in \code{implementation}.}
}
\value{
A list with: \item{sigma}{simulated correlation matrix.}
  \item{omega}{simulated precision matrix. Only returned if
  \code{output_matrices=TRUE}.} \item{theta}{adjacency matrix of the
  simulated graph. Only returned if \code{output_matrices=TRUE}.}
}
\description{
Simulates a correlation matrix. This is done in three steps with (i) the
simulation of an undirected graph encoding conditional independence, (ii) the
simulation of a (positive definite) precision matrix given the graph, and
(iii) the re-scaling of the inverse of the precision matrix.
}
\details{
In Step 1, the conditional independence structure between the
  variables is simulated. This is done using \code{\link{SimulateAdjacency}}.

  In Step 2, the precision matrix is simulated using
  \code{\link{SimulatePrecision}} so that (i) its nonzero entries correspond
  to edges in the graph simulated in Step 1, and (ii) it is positive definite
  (see \code{\link{MakePositiveDefinite}}).

  In Step 3, the covariance is calculated as the inverse of the precision
  matrix. The correlation matrix is then obtained by re-scaling the
  covariance matrix (see \code{\link[stats]{cov2cor}}).
}
\examples{
oldpar <- par(no.readonly = TRUE)
par(mar = rep(7, 4))

# Random correlation matrix
set.seed(1)
simul <- SimulateCorrelation(pk = 10)
Heatmap(simul$sigma,
  col = c("navy", "white", "darkred"),
  text = TRUE, format = "f", digits = 2,
  legend_range = c(-1, 1)
)

# Correlation matrix with homogeneous block structure
set.seed(1)
simul <- SimulateCorrelation(
  pk = c(5, 5),
  nu_within = 1,
  nu_between = 0,
  v_sign = -1,
  v_within = 1
)
Heatmap(simul$sigma,
  col = c("navy", "white", "darkred"),
  text = TRUE, format = "f", digits = 2,
  legend_range = c(-1, 1)
)

# Correlation matrix with heterogeneous block structure
set.seed(1)
simul <- SimulateCorrelation(
  pk = c(5, 5),
  nu_within = 0.5,
  nu_between = 0,
  v_sign = -1
)
Heatmap(simul$sigma,
  col = c("navy", "white", "darkred"),
  text = TRUE, format = "f", digits = 2,
  legend_range = c(-1, 1)
)

par(oldpar)

}
\seealso{
\code{\link{SimulatePrecision}}, \code{\link{MakePositiveDefinite}}

Other simulation functions: 
\code{\link{SimulateAdjacency}()},
\code{\link{SimulateClustering}()},
\code{\link{SimulateComponents}()},
\code{\link{SimulateGraphical}()},
\code{\link{SimulateRegression}()},
\code{\link{SimulateStructural}()}
}
\concept{simulation functions}
