\name{deconvolve}
\alias{deconvolve}
\title{Richardson-Lucy Deconvolution}
\description{
Performs a modified Richardson-Lucy iteration for the purpose of
estimating incidence from reported incidence or mortality, conditional
on a reporting probability and on a distribution of the time to
reporting.
}
\usage{
deconvolve(x, prob = 1, delay = 1,
           start, x.pad = 0, tol = 1, iter.max = 32L, complete = FALSE)
}
\arguments{
\item{x}{
  a numeric vector of length \code{n} giving the number of infections or
  deaths reported during \code{n} successive time intervals of equal
  duration.}
\item{prob}{
  a numeric vector of length \code{d+n}, \code{d=length(delay)-1}, such
  that \code{prob[d+i]} is the probability that an infection during
  interval \code{i} is eventually reported.  \code{prob} of length 1 is
  recycled.}
\item{delay}{
  a numeric vector of positive length such that \code{delay[j]} is the
  probability that an infection during interval \code{i} is reported
  during interval \code{i+j-1}, given that it is eventually reported.
  Hence \code{delay[j]} is the probability of a delay by \code{j-1}
  intervals and \code{d=length(delay)-1} is the maximum delay.
  \code{delay} need not sum to 1 but must not sum to 0.}
\item{start}{
  a numeric vector of length \code{d+n}, \code{d=length(delay)-1},
  giving a starting value for the iteration.  \code{start[d+i]}
  estimates the expected number of infections during interval \code{i}
  that are eventually reported.  If missing, then a starting value is
  generated by padding \code{x} on the left and right with \code{d-d0}
  and \code{d0} elements equal to \code{x.pad}, choosing
  \code{d0=which.max(delay)-1}.  Note that 0 is invariant under the
  iteration, hence it can be desirable to set \code{x.pad} to a small
  (relative to \code{max(x)}) positive number.}
\item{x.pad}{
  a non-negative number, used only when \code{start} is unset; see
  above.}
\item{tol}{
  a tolerance indicating a stopping condition; see the reference.  Set
  to 0 if you want to perform no fewer than \code{iter.max} iterations.}
\item{iter.max}{
  the maximum number of iterations.}
\item{complete}{
  a logical flag indicating if the result should preserve successive
  updates to \code{start}.}
}
\value{
A list with elements:
\item{value}{
  the result of updating \code{start} \code{iter} times then dividing by
  \code{prob}.  If \code{complete = TRUE}, then \code{value} is a
  \code{(d+n)}-by-\code{(1+iter)} matrix containing \code{start} and the
  \code{iter} successive updates, each divided by \code{prob}.}
\item{chisq}{
  the chi-squared statistics corresponding to \code{value}.}
\item{iter}{
  the number of iterations performed.}

\code{subset(value, start == 0)} is zero because zero is invariant under
the iteration.  If \code{delay} has \code{l} leading zeros and \code{t}
trailing zeros, then \code{head(value, t)} and \code{tail(value, l)} are
\code{NaN} due to divide-by-zero in the iteration.  (Conceptually,
\code{x} and \code{delay} provide no information about incidence during
those intervals.)
}
\details{
Do note that temporal alignment of \code{x} (length \code{n}) and
\code{y=deconvolve(x, \dots)$value} (length or number of rows
\code{d+n}) requires padding \code{x} on the left, as in
\code{cbind(x=c(rep(NA, d), x), y)}; see the examples.
}
%\seealso{}
\references{
Goldstein, E., Dushoff, J., Ma, J., Plotkin, J. B., Earn, D. J. D.,
& Lipsitch, M. (2009).
Reconstructing influenza incidence by deconvolution of daily mortality
time series.
\emph{Proceedings of the National Academy of Sciences U. S. A.},
\emph{106}(51), 21825-21829.
\doi{10.1073/pnas.0902958106}
}
\examples{
\dontshow{
## for R_DEFAULT_PACKAGES=NULL
library(   stats, pos = "package:base", verbose = FALSE)
library(graphics, pos = "package:base", verbose = FALSE)
library(   utils, pos = "package:base", verbose = FALSE)
}
##
## Example 1: simulation
##

set.seed(2L)
n <- 200L
d <- 50L
p <- 0.1
prob <- plogis(rlogis(d + n, location = qlogis(p), scale = 0.1))
delay <- c(0, diff(pgamma(0L:d, 12, 0.4)))

h <- function (x, a = 1, b = 1, c = 0) a * exp(-b * (x - c)^2)
ans <- floor(h(seq(-60, 60, length.out = d + n), a = 1000, b = 0.001))

x0 <- rbinom(d + n, ans, prob)
x <- tabulate(rep(1L:(d + n), x0) +
              sample(0L:d, size = sum(x0), replace = TRUE, prob = delay),
              d + n)[-(1L:d)]

str(D0 <- deconvolve(x, prob, delay, complete = FALSE))
str(D1 <- deconvolve(x, prob, delay, complete =  TRUE))

matplot(-(d - 1L):n,
        cbind(x0, c(rep(NA, d), x), prob * D0[["value"]], p * ans),
        type = c("p", "p", "p", "l"),
        col = c(1L, 1L, 2L, 4L), pch = c(16L, 1L, 16L, NA),
        lty = c(0L, 0L, 0L, 1L), lwd = c(NA, NA, NA, 3),
        xlab = "Time", ylab = "Count")
legend("topleft", NULL,
       c("actual", "actual+delay", "actual+delay+deconvolution", "p*h"),
       col = c(1L, 1L, 2L, 4L), pch = c(16L, 1L, 16L, NA),
       lty = c(0L, 0L, 0L, 1L), lwd = c(NA, NA, NA, 3),
       bty = "n")

plot(0L:D1[["iter"]], D1[["chisq"]],
     xlab = "Iterations", ylab = quote(chi^2))
abline(h = 1, lty = 2L)


##
## Example 2: application to pneumonia and influenza
##

data(pneumonia, package = "fastbeta")
x <- pneumonia[["series"]][["deaths"]]
delay <- pneumonia[["delay"]][["gpg"]]

n <- length(x)
d <- length(delay) - 1L
r <- 30L

D2 <- deconvolve(x = x, delay = delay, tol = 0, iter.max = r,
                 complete = TRUE)
stopifnot(D2[["iter"]] == r,
          identical(dim(D2[["value"]]), c(d + n, 1L + r)),
          length(D2[["chisq"]]) == 1L + r,
          min(D2[["chisq"]]) < 1)

## Subscript for the first, critical, and last values:
j2 <- c(1L, which.max(D2[["chisq"]] < 1), 1L + r)

matplot(x = seq(from = pneumonia[["series"]][1L, "date"] - d,
                by = 1, length.out = d + n),
        y = cbind(c(rep(NA, d), x), D2[["value"]][, j2]),
        type = "o",
        col = c(1L, 4L, 2L, 3L), pch = 1L, lty = 1L, lwd = 1,
        xlab = "1918", ylab = "deaths")
legend("topleft", NULL,
       c("observed", sprintf("after \%d iterations", j2 - 1L)),
       col = c(1L, 4L, 2L, 3L), pch = 1L, lty = 1L, lwd = 1,
       bty = "n")
}
