% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hrf.R
\name{HRF_objects}
\alias{HRF_objects}
\alias{HRF_SPMG1}
\alias{HRF_SPMG2}
\alias{HRF_SPMG3}
\alias{HRF_GAMMA}
\alias{HRF_GAUSSIAN}
\alias{HRF_BSPLINE}
\alias{HRF_FIR}
\title{Pre-defined Hemodynamic Response Function Objects}
\usage{
HRF_GAMMA(t, shape = 6, rate = 1)

HRF_GAUSSIAN(t, mean = 6, sd = 2)

HRF_SPMG1(t, P1 = 5, P2 = 15, A1 = 0.0833)

HRF_SPMG2(t)

HRF_SPMG3(t)

HRF_BSPLINE(t)

HRF_FIR(t)
}
\arguments{
\item{t}{Numeric vector of time points (in seconds) at which to evaluate the HRF}

\item{shape, rate}{Parameters for gamma distribution HRF (default: shape=6, rate=1)}

\item{mean, sd}{Parameters for Gaussian HRF (default: mean=6, sd=2)}

\item{P1, P2}{Shape parameters for SPM canonical HRF (default: P1=5, P2=15)}

\item{A1}{Amplitude parameter for SPM canonical HRF (default: 0.0833)}
}
\value{
When called as functions, return numeric vectors or matrices of HRF values.
When used as objects, they are HRF objects with class \code{c("HRF", "function")}.
}
\description{
A collection of pre-defined HRF objects for common fMRI analysis scenarios.
These objects can be used directly in model specifications or as templates
for creating custom HRFs.
}
\section{Canonical HRFs}{

\describe{
  \item{\code{HRF_SPMG1}}{SPM canonical HRF (single basis function)}
  \item{\code{HRF_SPMG2}}{SPM canonical HRF with temporal derivative (2 basis functions)}
  \item{\code{HRF_SPMG3}}{SPM canonical HRF with temporal and dispersion derivatives (3 basis functions)}
  \item{\code{HRF_GAMMA}}{Gamma function-based HRF}
  \item{\code{HRF_GAUSSIAN}}{Gaussian function-based HRF}
}
}

\section{Flexible Basis Sets}{

\describe{
  \item{\code{HRF_BSPLINE}}{B-spline basis HRF (5 basis functions)}
  \item{\code{HRF_FIR}}{Finite Impulse Response (FIR) basis HRF (12 basis functions)}
}
}

\section{Creating Custom Basis Sets}{

The pre-defined objects above have fixed numbers of basis functions. To create
basis sets with custom parameters (e.g., different numbers of basis functions),
use one of these approaches:

\strong{Using getHRF():}
\itemize{
  \item \code{getHRF("fir", nbasis = 20)} - FIR basis with 20 functions
  \item \code{getHRF("bspline", nbasis = 10, span = 30)} - B-spline with 10 functions
  \item \code{getHRF("fourier", nbasis = 7)} - Fourier basis with 7 functions
  \item \code{getHRF("daguerre", nbasis = 5, scale = 3)} - Daguerre basis
}

\strong{Using generator functions directly:}
\itemize{
  \item \code{hrf_fir_generator(nbasis = 20, span = 30)}
  \item \code{hrf_bspline_generator(nbasis = 10, span = 30)}
  \item \code{hrf_fourier_generator(nbasis = 7, span = 24)}
  \item \code{hrf_daguerre_generator(nbasis = 5, scale = 3)}
}
}

\section{Usage}{

All HRF objects can be:
\itemize{
  \item Called as functions with time argument: \code{HRF_SPMG1(t)}
  \item Used in model specifications: \code{hrf(condition, basis = HRF_SPMG1)}
  \item Evaluated with \code{evaluate()} method
  \item Combined with decorators like \code{lag_hrf()} or \code{block_hrf()}
}
}

\examples{
# Evaluate HRFs at specific time points
times <- seq(0, 20, by = 0.5)

# Single basis canonical HRF
canonical_response <- HRF_SPMG1(times)
plot(times, canonical_response, type = "l", main = "SPM Canonical HRF")

# Multi-basis HRF with derivatives
multi_response <- HRF_SPMG3(times)  # Returns 3-column matrix
matplot(times, multi_response, type = "l", main = "SPM HRF with Derivatives")

# Gamma and Gaussian HRFs
gamma_response <- HRF_GAMMA(times)
gaussian_response <- HRF_GAUSSIAN(times)

# Compare different HRF shapes
plot(times, canonical_response, type = "l", col = "blue", 
     main = "HRF Comparison", ylab = "Response")
lines(times, gamma_response, col = "red")
lines(times, gaussian_response, col = "green")
legend("topright", c("SPM Canonical", "Gamma", "Gaussian"), 
       col = c("blue", "red", "green"), lty = 1)

# Create custom FIR basis with 20 bins
custom_fir <- getHRF("fir", nbasis = 20, span = 30)
fir_response <- evaluate(custom_fir, times)
matplot(times, fir_response, type = "l", main = "Custom FIR with 20 bins")

# Create custom B-spline basis  
custom_bspline <- hrf_bspline_generator(nbasis = 8, span = 25)
bspline_response <- evaluate(custom_bspline, times)
matplot(times, bspline_response, type = "l", main = "Custom B-spline with 8 basis functions")

}
\seealso{
\code{\link{evaluate.HRF}} for evaluating HRF objects,
\code{\link{gen_hrf}} for creating HRFs with decorators,
\code{\link{list_available_hrfs}} for listing all HRF types,
\code{\link{getHRF}} for creating HRFs by name with custom parameters,
\code{\link{hrf_fir_generator}}, \code{\link{hrf_bspline_generator}}, 
\code{\link{hrf_fourier_generator}}, \code{\link{hrf_daguerre_generator}} 
for creating custom basis sets directly

Other hrf: 
\code{\link{deriv}()},
\code{\link{penalty_matrix}()}
}
\concept{hrf}
