% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_numbers.R
\name{format_engr}
\alias{format_engr}
\title{Format engineering notation}
\usage{
format_engr(
  x,
  digits = 4,
  ...,
  omit_power = c(-1, 2),
  set_power = NULL,
  delim = formatdown_options("delim"),
  size = formatdown_options("size"),
  decimal_mark = formatdown_options("decimal_mark"),
  small_mark = formatdown_options("small_mark"),
  small_interval = formatdown_options("small_interval"),
  whitespace = formatdown_options("whitespace")
)
}
\arguments{
\item{x}{Number or numbers to be formatted. Can be a single number, a vector,
or a column of a data frame.}

\item{digits}{Integer from 1 through 20 that controls the number of
significant digits in printed numeric values. Passed to \code{signif()}.
Default is 4.}

\item{...}{Not used for values; forces subsequent arguments to be
referable only by name.}

\item{omit_power}{Numeric vector \code{c(p, q)} with \code{p <= q}, specifying
the range of exponents over which power-of-ten notation is omitted in
either scientific or engineering format. Default is
\code{c(-1, 2)}. If a single value is assigned, i.e., \code{omit_power = p}, the
argument is interpreted as \code{c(p, p)}. If \code{NULL} or \code{NA}, all elements
are formatted in power-of-ten notation. Argument is overridden by
specifying \code{set_power} or decimal notation.}

\item{set_power}{Integer, length 1. Formats all values in \code{x} with the same
power-of-ten exponent. Default NULL. Overrides \code{format} and \code{omit_power}
arguments.}

\item{delim}{Character, length 1 or 2, to define the left and right math
markup delimiters. The default setting, \code{delim = "$"}, produces
left and right delimiters \verb{$...$}. The alternate built-in
setting, \code{delim = "\\("}, produces left and right delimiters
\verb{\\\\( ... \\\\)}. Custom delimiters can be assigned in a vector of
length 2 with left and right delimiter symbols, e.g.,
\verb{c("\\\[", "\\\]")}. Special characters typically must be escaped.}

\item{size}{Character, length 1, to assign a font size. If not empty, adds
a font size macro to the markup inside the math delimiters. Possible
values are \code{"scriptsize"}, \code{"small"}, \code{"normalsize"}, \code{"large"}, and
\code{"huge"}. One may also assign the equivalent LaTeX-style markup itself,
e.g., \code{"\\\\scriptsize"}, \code{"\\\\small"}, etc. Default is NULL.}

\item{decimal_mark}{Character, length 1, to assign the decimal marker.
Possible values are a period \code{"."} (default) or a comma \code{","}. Passed
to \code{formatC(decimal.mark)}.}

\item{small_mark}{Character, length 1, used as the mark between every
\code{small_interval} number of digits to the right of the decimal marker to
improve readability. Possible values are empty \code{""} (default) or
\code{"thin"} to produce a LaTeX-style thin, horizontal space. One may also
assign the thin-space markup itself \code{"\\\\\\\\,"}.
Passed to \code{formatC(small.mark)}.}

\item{small_interval}{Integer, length 1, that defines the number of digits
(default 5) in groups separated by \code{small_mark}. Passed to
\code{formatC(small.interval)}.}

\item{whitespace}{Character, length 1, to define the LaTeX-style
math-mode macro to preserve a horizontal space between words of text or
between physical-unit abbreviations when formatting numbers of class
"units". Default is \code{"\\\\\\\\>"}. Alternatives include \code{"\\\\\\\\:"} or
"\verb{\\\\\\\\ }".}
}
\value{
A character vector in which numbers are formatted in power-of-ten
notation in engineering form and delimited for rendering as inline equations
in an R markdown document.
}
\description{
Convert a numeric vector to a character vector in which the numbers are
formatted in power-of-ten notation in engineering form and delimited for
rendering as inline equations in an R markdown document.
}
\details{
In engineering notation, all exponents are multiples of three.
\code{format_engr()} is a wrapper for the more general function \code{format_numbers()}.
Where defaults are defined by \code{formatdown_options()}, users may reassign
the arguments locally in the function call or globally using
\code{formatdown_options()}.

Arguments after the dots (\code{...}) must be referred to by name.
}
\examples{
# input: single number
x <- 6.0221E+23
format_numbers(x)

# input: units class
x <- 103400
units(x) <- "N m2 C-2"
format_numbers(x)

# input: vector
data("metals", package = "formatdown")
x <- metals$dens
format_numbers(x)

# significant digits
x <- 9.75358e+5
format_numbers(x, 2)
format_numbers(x, 3)
format_numbers(x, 4)

# format & wrappers: format_engr(), format_sci(), format_dcml()
x <- 6.0221E+23
format_numbers(x, format = "engr")
format_engr(x)

format_numbers(x, format = "sci")
format_sci(x)

x <- 103400
format_numbers(x, format = "dcml")
format_dcml(x)

# input: data frame
x <- metals[, c("thrm_exp", "thrm_cond")]
as.data.frame(apply(x, 2, format_sci, digits = 3))

# omit_power
x <- 103400
format_sci(x, omit_power = c(-1, 2)) # default
format_sci(x, omit_power = c(-1, 5))
format_sci(x, omit_power = 5) # equivalent to omit_power = c(5, 5)
x <- 1.2
format_sci(x, omit_power = NULL)

# set_power
format_sci(x, set_power = NULL) # default
format_sci(x, set_power = 3)

# set_power overrides format
x <- 6.0221E+23
format_engr(x)
format_engr(x, set_power = 24L)
format_sci(x)
format_sci(x, set_power = 24L)

# set_power overrides omit_power
x <- 101300
format_sci(x, omit_power = 5)
format_sci(x, omit_power = 5, set_power = 2)
format_sci(x, omit_power = 2)
format_sci(x, omit_power = 2, set_power = 2)

# decimal format ignores set_power
x <- 103400
format_numbers(x, format = "dcml")
format_numbers(x, format = "dcml", set_power = 3)

}
\seealso{
Other format_*: 
\code{\link{format_dcml}()},
\code{\link{format_numbers}()},
\code{\link{format_sci}()},
\code{\link{format_text}()}
}
\concept{format_*}
