% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fuzzy_topological_relations.R
\name{spa_set_classification}
\alias{spa_set_classification}
\title{Set a new classification for fuzzy topological relationships}
\usage{
spa_set_classification(classes, mfs)
}
\arguments{
\item{classes}{A character vector containing linguistic values that characterizes different situations of fuzzy topological relationships.}

\item{mfs}{A vector of membership functions with domain in [0, 1].}
}
\value{
No return values, called for side effects.
}
\description{
\code{spa_set_classification()} configures a new set of linguistic values and corresponding membership functions to be used by fuzzy topological relationships.
}
\details{
The \code{spa_set_classification()} function replaces the default linguistic values employed by fuzzy topological relationships.
Each membership function \emph{i} of the parameter \code{mfs} represents the class \emph{i} of the parameter \code{classes}.
The length of these parameters must to be equal.
}
\examples{
\dontrun{
library(tibble)
library(sf)

set.seed(456)

# Generating some random points to create pgeometry objects by using spa_creator()
tbl = tibble(x = runif(10, min= 0, max = 30), 
             y = runif(10, min = 0, max = 30), 
             z = runif(10, min = 0, max = 50))

# Getting the convex hull on the points to clip plateau region objects during their constructions
pts <- st_as_sf(tbl, coords = c(1, 2))
ch <- st_convex_hull(do.call(c, st_geometry(pts)))

pregions <- spa_creator(tbl, base_poly = ch, fuzz_policy = "fcp", k = 2)

plot(pregions$pgeometry[[1]])
plot(pregions$pgeometry[[2]])

# Showing results for spa_overlap() by considering default list of classes
spa_overlap(pregions$pgeometry[[1]], pregions$pgeometry[[2]], ret = "list")
}
# Changing the default classification 
trap_mf <- function(a, b, c, d) {
  function(x) {
    pmax(pmin((x - a)/(b - a), 1, (d - x)/(d - c), na.rm = TRUE), 0)
  }
}

classes <- c("superficially", "moderately", "completely")
superficially <- trap_mf(0, 0.2, 0.4, 0.6)
moderately <- trap_mf(0.4, 0.6, 0.8, 1)
completely <- trap_mf(0.6, 0.8, 1, 1)

spa_set_classification(classes, c(superficially, moderately, completely))
\dontrun{
# Now the fuzzy topological relationships will use the new classification
spa_overlap(pregions$pgeometry[[1]], pregions$pgeometry[[2]], ret = "list")
}
}
\references{
\href{https://onlinelibrary.wiley.com/doi/10.1111/tgis.13044}{Carniel, A. C.; Venâncio, P. V. A. B; Schneider, M. fsr: An R package for fuzzy spatial data handling. Transactions in GIS, vol. 27, no. 3, pp. 900-927, 2023.}

Underlying concepts and formal definitions of spatial plateau topological relationships and fuzzy topological relationships are respectively introduced in:
\itemize{
\item \href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
\item \href{https://ieeexplore.ieee.org/document/7737976}{Carniel, A. C.; Schneider, M. A Conceptual Model of Fuzzy Topological Relationships for Fuzzy Regions. In Proceedings of the 2016 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2016), pp. 2271-2278, 2016.}
}
}
