
#' Compute the functional diversity indices and community weighted means.
#'
#' @param fwdata \code{list} A list from the \code{fw_fetchdata} function is the only data type accepted.
#' @param sitesdata \code{dataframe} Dataframe with sites for which functional diversity and
#'        community weighted means indices will be computed for.
#' @param species \code{string} The species column in the sitesdata.
#' @param sites  \code{string} The sites column in the sitesdata. If the sites are not provided by the geometry column, then
#'        sites can be auto generated by clustering closely separated records into sites.
#' @param dist \code{integer} The distance used to cluster points as sites. The default is 1000.
#' @param abund \code{string} Column with abundance data. If not provided, then only functional richness can be computed.
#' @param FD \code{logical} To indicate whether to compute functional richness or community weighted means. The default is TRUE.
#' @param dummy \code{logical} To form fuzzy code for the categorical traits. Default is TRUE.
#'
#' @return dataframe
#' @export
#'
#' @examples
#'
#' \dontrun{
#'
#' set.seed(1135)
#' data('speciesdata')
#' speciesdata$abundance <- rnorm(n = nrow(speciesdata), 4.3, 1.2)
#'
#' spgeo1 <- speciesdata |>
#'  sf::st_as_sf(coords = c('decimalLongitude', 'decimalLatitude'), crs = sf::st_crs(4326))
#'
#' sptraits <- fw_fetchdata(data = speciesdata,
#'                         ecoparams = c('rheophily habitat', 'spawning habitat',
#'                                       'feeding diet adult'),
#'                         taxonomic_column = 'scientificName',
#'                         organismgroup = 'fi')
#' #functional richness
#' testdata5 <- fw_fdcompute(fwdata = sptraits,
#'                          sitesdata = speciesdata,
#'                          sites = 'waterBody',
#'                          species = 'scientificName',
#'                          FD = TRUE)
#'
#' }
#'
fw_fdcompute <- function(fwdata, sitesdata, species, sites = NULL, dist = 1000,
                          abund = NULL,
                          FD= TRUE, dummy = TRUE){

  dd <- fwdata$ecodata |>
    dplyr::select(.data$Parameter, .data$Taxonname, .data$CategoryName) |>
    tidyr::pivot_wider(names_from = .data$Parameter, values_from = .data$CategoryName,
                       values_fn =  dplyr::first)|>
    tibble::column_to_rownames('Taxonname') |>
    stats::na.omit()

  if(!inherits(sitesdata, 'sf')& is.null(sites))stop("For sites data with no coordinates, provide the sites column.")

  if(inherits(sitesdata, 'sf') && is.null(sites)){

    sdata <- fw_geosites(x= sitesdata, dist = dist) |> sf::st_drop_geometry()
    sx <- 'sites'

  }  else{
    if(inherits(sitesdata, 'sf') && !is.null(sites)) sitesdata <- sitesdata |> sf::st_drop_geometry()

    if(isFALSE(sites%in%colnames(sitesdata))) stop('The ', sites, ' column must be found in the ', deparse(substitute(sitesdata)), ' data.')

    sdata <- sitesdata
    sx <- sites
  }

  sdata <- sdata |> dplyr::filter(.data[[species]]%in%rownames(dd))
  value <- NULL
  if(is.null(abund)){
    comm <- sdata |>  dplyr::select({{species}}, {{sx}})|>
      dplyr:: mutate(value = 1) |>
      dplyr::distinct(.keep_all = TRUE)|>
      tidyr::pivot_wider(names_from = species, values_from = value, values_fill = 0)|>
      tibble::column_to_rownames(sx)
  }else{
    comm <- sdata |>  dplyr::select({{species}}, {{sx}}, {{abund}})|>
      dplyr::group_by(.data[[species]], .data[[sx]]) |>
      dplyr::summarise(abund = mean(.data[[abund]]), .groups = 'drop') |>
      tidyr::pivot_wider(names_from = species, values_from = abund, values_fill = 0)|>
      tibble::column_to_rownames(sx)
  }

  if(isTRUE(FD)){

    traitfuzzy <- fuzzy_codes(trait = dd, FD = FD)

    traitsdata <- traitfuzzy[colnames(comm), ]

    ab <- as.matrix(comm)

    if(is.null(abund)){

      frdata <- suppressWarnings(FD::dbFD(x = traitsdata, a = ab,calc.CWM = FALSE,
                                          calc.FRic = TRUE, stand.FRic = TRUE,
                                          messages = FALSE))
      out <- as.data.frame(frdata[c("FRic", "nbsp")]) |> dplyr::rename(SRic = "nbsp")
    }else{
      frdata <- suppressWarnings(FD::dbFD(x = traitsdata, a = ab,calc.CWM = TRUE,
                                          calc.FRic = TRUE, messages = FALSE,
                                          stand.FRic = TRUE))
      out <-  as.data.frame(frdata[c("FRic",  "nbsp", "FEve", "FDiv", "FDis", "RaoQ")]) |>

        dplyr::rename(SRic = "nbsp")
    }

  }else{

    if(isTRUE(dummy)){

      traitfuzzy <- fuzzy_codes(trait = dd , FD = FD)

      traitsdata <- traitfuzzy[colnames(comm), ]

      ab <- as.matrix(comm)

      cwm <- FD::functcomp(x = traitsdata, a = ab, CWM.type = "all")

      out <- cwm[, !grepl("_0$", colnames(cwm))]

      colnames(out) <- sub("_1$", "", colnames(out))

      out
    }else{
      trds <- as.matrix(dd)

      ab <- as.matrix(comm)

      ix <- match(rownames(trds), colnames(ab))

      abbv <- ab[, ix]

      out <- FD::functcomp(x = trds, abbv, CWM.type = "all")
    }
  }
  out<- tibble::rownames_to_column(out, var = "site")

}
