#' @details This function simulates balanced or unbalanced RSS sampling for proportions. The length of the sample allocation vector (nsamp) must match the set size (H). The p parameter adjusts the true population proportion. The simulations assumes perfect ranking.
#' @title Generate example ranked set samples for proportions
#' @name rss.prop.simulation
#' @description The rss.prop.simulation function generates ranked set samples for proportions by simulating data with options to adjust the population proportion (p).
#'
#' @param H The RSS set size.
#' @param nsamp A numeric vector specifying the sample allocation for each stratum.
#' @param p The true population proportion.
#'
#' @return
#' \item{rank}{The rank information assigned to each sample.}
#' \item{y}{The generated ranked set samples (binary values) based on the population proportion.}
#' @examples
#' ## Balanced RSS with a set size 3 and equal sample sizes of 6 for each stratum,
#' ## using perfect ranking with true proportion of 0.6.
#' rss.prop.data=rss.prop.simulation(H=3,nsamp=c(6,6,6),p=0.6)
#'
#' ## Unbalanced RSS with a set size 3 and different sample sizes of 6, 10, and 8 for each stratum,
#' ## using perfect ranking with true proportion of 0.2.
#' rss.prop.data=rss.prop.simulation(H=3,nsamp=c(6,10,8),p=0.2)
#'
#' # Check the structure of the RSS data
#' colnames(rss.prop.data) # Should include "y" and "rank"
#' head(rss.prop.data$y)
#' head(rss.prop.data$rank)
#'
#' @export
rss.prop.simulation <- function(H, nsamp, p)
{
  n=sum(nsamp)
  data=matrix(0,n,2)
  if(H != length(nsamp)) stop("Set size are different with the length of sample allocations", call. = F)
  if(p <0 || p >1) stop("Invalid value for p. It must be between 0 and 1.")

  for(h in (1:H)){
    for(i in (1:nsamp[h])){
      k=i
      if(h>1){
        k=i+sum(nsamp[1:(h-1)])
      }
      tdata1=stats::runif(H) # uniform distribution for ranking
      rtdata1=rank(tdata1,ties.method='first') # ranking
      tdata2=ifelse(tdata1>(1-p),1,0) # convert proportion into binary

      data[k,1]=h
      data[k,2]=tdata2[rtdata1==h]
    }
  }
  colnames(data) <- c("rank","y")
  data<-as.data.frame(data)
  return(data)
}
