% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat_brace.R
\name{stat_brace}
\alias{stat_brace}
\title{create curly braces as a layer in ggplot}
\usage{
stat_brace(
  mapping = NULL,
  data = NULL,
  ...,
  rotate = 0,
  width = NULL,
  mid = NULL,
  outside = TRUE,
  distance = NULL,
  outerstart = NULL,
  bending = NULL,
  show.legend = FALSE,
  inherit.aes = TRUE,
  discreteAxis = FALSE,
  bracketType = "curly",
  npoints = 100
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}'s \code{params} argument. These
arguments broadly fall into one of 4 categories below. Notably, further
arguments to the \code{position} argument, or aesthetics that are required
can \emph{not} be passed through \code{...}. Unknown arguments that are not part
of the 4 categories below are ignored.
\itemize{
\item Static aesthetics that are not mapped to a scale, but are at a fixed
value and apply to the layer as a whole. For example, \code{colour = "red"}
or \code{linewidth = 3}. The geom's documentation has an \strong{Aesthetics}
section that lists the available options. The 'required' aesthetics
cannot be passed on to the \code{params}. Please note that while passing
unmapped aesthetics as vectors is technically possible, the order and
required length is not guaranteed to be parallel to the input data.
\item When constructing a layer using
a \verb{stat_*()} function, the \code{...} argument can be used to pass on
parameters to the \code{geom} part of the layer. An example of this is
\code{stat_density(geom = "area", outline.type = "both")}. The geom's
documentation lists which parameters it can accept.
\item Inversely, when constructing a layer using a
\verb{geom_*()} function, the \code{...} argument can be used to pass on parameters
to the \code{stat} part of the layer. An example of this is
\code{geom_area(stat = "density", adjust = 0.5)}. The stat's documentation
lists which parameters it can accept.
\item The \code{key_glyph} argument of \code{\link[ggplot2:layer]{layer()}} may also be passed on through
\code{...}. This can be one of the functions described as
\link[ggplot2:draw_key]{key glyphs}, to change the display of the layer in the legend.
}}

\item{rotate}{number in degrees. Defines where the brace points to: 0=up (default), 90=right, 180=down, 270=left}

\item{width}{number. Distance from the brace's start to its tip. If NULL (default), will be determined by data.}

\item{mid}{number from 0.25 to 0.75. Position of the pointer within the brace space. If NULL (default), will be determined by data.}

\item{outside}{boolean. If TRUE (default), brace is next to the data area. If FALSE, brace is inside the data area}

\item{distance}{number. Space between the brace and the nearest data point. If NULL (default), will be determined by data.}

\item{outerstart}{number. If not NULL, overwrites distance and sets all braces to the same origin}

\item{bending}{number from 0 to 0.5. Determines bend of the brace curves (0=rectangular). If NULL (default), will be determined by data. If too high, values will result in zick-zack lines}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}

\item{discreteAxis}{boolean. Set to TRUE if the axis along which the brace expands is discrete (often true for bar graphs)}

\item{bracketType}{text choice. Either "curly" (default) or "square"}

\item{npoints}{integer. Number of points generated for the brace curves. Will be rounded to be a multiple of 4 for calculation purposes.}
}
\value{
ggplot2 layer object (geom_path) that can directly be added to a ggplot2 object.
}
\description{
Imports:
ggplot2
}
\examples{
library(ggbrace)
library(ggplot2)
data(iris)

# regular braces
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace()

 # rotated braces
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(rotate = 90)

 # braces inside the given coordinates
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(outside = FALSE)

 # braces with a defined distance from their data points
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(distance = 2)

 # braces starting at a defined point
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(outerstart = 5)

 # braces starting at a defined point and with defined width
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(outerstart = 5, width = 1)

 # braces starting at a defined point and with defined width and defined curve bending
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(outerstart = 5, width = 1, bending = 0.1)

 # braces outside of the plotting area
ggplot(iris, aes(x=Sepal.Length, y=Sepal.Width, color=Species, label=Species)) +
 geom_point() +
 stat_brace(outerstart = 4.5) +
 coord_cartesian(y=range(iris$Sepal.Width), clip = "off") +
 theme(plot.margin = unit(c(0.25, 0.11, 0.11, 0.11), units="npc"))

 # braces with discrete axes
 df <- iris
 df$Group <- substring(iris$Species,1,1)
 ggplot(df, aes(x=Species, y=Sepal.Length, group=Group)) +
   geom_jitter() +
   stat_brace(discreteAxis=TRUE)


}
