% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/density.R
\name{density_bounded}
\alias{density_bounded}
\title{Bounded density estimator using the reflection method}
\usage{
density_bounded(
  x,
  weights = NULL,
  n = 501,
  bandwidth = "dpi",
  adjust = 1,
  kernel = "gaussian",
  trim = TRUE,
  bounds = c(NA, NA),
  bounder = "cdf",
  adapt = 1,
  na.rm = FALSE,
  ...,
  range_only = FALSE
)
}
\arguments{
\item{x}{<\link{numeric}> Sample to compute a density estimate for.}

\item{weights}{<\link{numeric} | \link{NULL}> Optional weights to apply to \code{x}.}

\item{n}{<scalar \link{numeric}> The number of grid points to evaluate the density estimator at.}

\item{bandwidth}{<scalar \link{numeric} | \link{function} | \link[=character]{string}>
Bandwidth of the density estimator. One of:
\itemize{
\item a numeric: the bandwidth, as the standard deviation of the kernel
\item a function: a function taking \code{x} (the sample) and returning the bandwidth
\item a string: the suffix of the name of a function starting with \code{"bandwidth_"} that
will be used to determine the bandwidth. See \link{bandwidth} for a list.
}}

\item{adjust}{<scalar \link{numeric}> Value to multiply the bandwidth of the density estimator by. Default \code{1}.}

\item{kernel}{<\link[=character]{string}> The smoothing kernel to be used. This must partially
match one of \code{"gaussian"}, \code{"rectangular"}, \code{"triangular"}, \code{"epanechnikov"},
\code{"biweight"}, \code{"cosine"}, or \code{"optcosine"}. See \code{\link[stats:density]{stats::density()}}.}

\item{trim}{<scalar \link{logical}> Should the density estimate be trimmed to the range of the data? Default \code{TRUE}.}

\item{bounds}{<length-2 \link{numeric}> Min and max bounds. If a bound is \code{NA}, then
that bound is estimated from the data using the method specified by \code{bounder}.}

\item{bounder}{<\link{function} | \link[=character]{string}> Method to use to find missing
(\code{NA}) \code{bounds}. A function that
takes a numeric vector of values and returns a length-2 vector of the estimated
lower and upper bound of the distribution. Can also be a string giving the
suffix of the name of such a function that starts with \code{"bounder_"}. Useful
values include:
\itemize{
\item \code{"cdf"}: Use the CDF of the the minimum and maximum order statistics of the
sample to estimate the bounds. See \code{\link[=bounder_cdf]{bounder_cdf()}}.
\item \code{"cooke"}: Use the method from Cooke (1979); i.e. method 2.3 from Loh (1984).
See \code{\link[=bounder_cooke]{bounder_cooke()}}.
\item \code{"range"}: Use the range of \code{x} (i.e the \code{min} or \code{max}). See \code{\link[=bounder_range]{bounder_range()}}.
}}

\item{adapt}{<positive \link{integer}> (\strong{very experimental}) The name and interpretation of this argument
are subject to change without notice. If \code{adapt > 1}, uses
an adaptive approach to calculate the density. First, uses the
adaptive bandwidth algorithm of Abramson (1982) to determine local (pointwise)
bandwidths, then groups these bandwidths into \code{adapt} groups, then calculates
and sums the densities from each group. You can set this to a very large number
(e.g. \code{Inf}) for a fully adaptive approach, but this will be very slow; typically
something around 100 yields nearly identical results.}

\item{na.rm}{<scalar \link{logical}> Should missing (\code{NA}) values in \code{x} be removed?}

\item{...}{Additional arguments (ignored).}

\item{range_only}{<scalar \link{logical}> If \code{TRUE}, the range of the output of this density estimator
is computed and is returned in the \verb{$x} element of the result, and \code{c(NA, NA)}
is returned in \verb{$y}. This gives a faster way to determine the range of the output
than \code{density_XXX(n = 2)}.}
}
\value{
An object of class \code{"density"}, mimicking the output format of
\code{\link[stats:density]{stats::density()}}, with the following components:
\itemize{
\item \code{x}: The grid of points at which the density was estimated.
\item \code{y}: The estimated density values.
\item \code{bw}: The bandwidth.
\item \code{n}: The sample size of the \code{x} input argument.
\item \code{call}: The call used to produce the result, as a quoted expression.
\item \code{data.name}: The deparsed name of the \code{x} input argument.
\item \code{has.na}: Always \code{FALSE} (for compatibility).
\item \code{cdf}: Values of the (possibly weighted) empirical cumulative distribution
function at \code{x}. See \code{\link[=weighted_ecdf]{weighted_ecdf()}}.
}

This allows existing methods for density objects, like \code{\link[=print]{print()}} and \code{\link[=plot]{plot()}}, to work if desired.
This output format (and in particular, the \code{x} and \code{y} components) is also
the format expected by the \code{density} argument of the \code{\link[=stat_slabinterval]{stat_slabinterval()}}
and the \code{\link[=smooth_density]{smooth_}} family of functions.
}
\description{
Bounded density estimator using the reflection method.

Supports \link[=auto_partial]{automatic partial function application} with
\link[=waiver]{waived arguments}.
}
\examples{
library(distributional)
library(dplyr)
library(ggplot2)

# For compatibility with existing code, the return type of density_bounded()
# is the same as stats::density(), ...
set.seed(123)
x = rbeta(5000, 1, 3)
d = density_bounded(x)
d

# ... thus, while designed for use with the `density` argument of
# stat_slabinterval(), output from density_bounded() can also be used with
# base::plot():
plot(d)

# here we'll use the same data as above, but pick either density_bounded()
# or density_unbounded() (which is equivalent to stats::density()). Notice
# how the bounded density (green) is biased near the boundary of the support,
# while the unbounded density is not.
data.frame(x) \%>\%
  ggplot() +
  stat_slab(
    aes(xdist = dist), data = data.frame(dist = dist_beta(1, 3)),
    alpha = 0.25
  ) +
  stat_slab(aes(x), density = "bounded", fill = NA, color = "#d95f02", alpha = 0.5) +
  stat_slab(aes(x), density = "unbounded", fill = NA, color = "#1b9e77", alpha = 0.5) +
  scale_thickness_shared() +
  theme_ggdist()

# We can also supply arguments to the density estimators by using their
# full function names instead of the string suffix; e.g. we can supply
# the exact bounds of c(0,1) rather than using the bounds of the data.
data.frame(x) \%>\%
  ggplot() +
  stat_slab(
    aes(xdist = dist), data = data.frame(dist = dist_beta(1, 3)),
    alpha = 0.25
  ) +
  stat_slab(
    aes(x), fill = NA, color = "#d95f02", alpha = 0.5,
    density = density_bounded(bounds = c(0,1))
  ) +
  scale_thickness_shared() +
  theme_ggdist()
}
\references{
Cooke, P. (1979). Statistical inference for bounds of random variables.
\emph{Biometrika} 66(2), 367--374. \doi{10.1093/biomet/66.2.367}.

Loh, W. Y. (1984). Estimating an endpoint of a distribution with resampling
methods. \emph{The Annals of Statistics} 12(4), 1543--1550. \doi{10.1214/aos/1176346811}
}
\seealso{
Other density estimators: 
\code{\link{density_histogram}()},
\code{\link{density_unbounded}()}
}
\concept{density estimators}
